# Authentication
Authentication on the INJECT platform is fundamental to ensuring secure access to its features and resources. Leveraging Django's built-in session system, the platform utilizes cookies as a means to authenticate client requests.

## Authentication Process Overview

The authentication process involves the following steps:
``` mermaid
sequenceDiagram
  Client->>Server: Get backend version
  Server->>Client: Responds with backend version and SET_COOKIE
```
```
1. Client -> Server: Get the backend version from the REST API. (No cookies are set yet.)
2. Client <- Server: Responds with backend version and SET_COOKIE 
with new _sessionid_ and _csrf_ cookie.
```

In this state, the server established a session with the client. The client did not authenticate yet, and thus the session attribute _user_ is set to _AnonymousUser_. All further requests from the client should now contain the _sessionid_ and _csrf_ cookie together with the _X-csrftoken_ header of requests (most importantly for safe login!).
The client can now make any request to endpoints, which do not enforce authenticated requesters. On every request, the match of _csrf_ cookie and _X-csrftoken_ header is checked.
The next important step in our schema is **login**:

``` mermaid
sequenceDiagram
  Client->>Server: Sends login request on GraphQL login endpoint
  Server->>Client: Server generates new session
```
```
3. Client -> Server: Sends a login request on the GraphQL login endpoint 
containing the username and password (and most importantly, 
the _sesisonid_ and _csrf_ cookie with _X-csrftoken_ header set to the 
value of the _csrf_ cookie to prevent cross-site request forgery).
4. Client <- Server: If the provided credentials were correct, 
the server generated a new session with attribute _user_ set 
to User model of user who successfully authenticated themselves and 
responded with SET_COOKIE containing the new _sessionid_ cookie 
and the new _csrf_ cookie.
```

The client is expected to drop the old _sessionid_ and _csrf_ cookie and replace them with the new ones. Also, every request from now on should contain _X-csrftoken_ value of the new _csrf_ cookie.

For further information, refer to the official documentation on:

* [Django Sessions](https://docs.djangoproject.com/en/2.0/topics/http/sessions/)
* [CSRF Protection](https://docs.djangoproject.com/en/2.0/ref/csrf/)

### Basic authentication
For the purposes of more convenient development, basic authentication is also supported according to the [RFC 2617](https://datatracker.ietf.org/doc/html/rfc2617#section-2) standard.

## Authorization
For authorization processes, the RBAC (Role-based access control) is supplemented with a modified ACL (access control list)  

Three authorization roles (also called groups to not be mistaken for in-game roles) are present:  
- **ADMIN**  
  - has implicit access to all resources and the platform (exercises, definitions, users, etc...)  
  - role should be assigned only to people that really need it (maintainers, deployers of the platform), and the number of people with this role should be    kept to a minimum  
  - can execute every action that a TRAINEE or INSTRUCTOR can  
- **INSTRUCTOR**  
  - can access only exercises and definitions where he/she was assigned (or which he/she has created and was not removed from them)  
  - isntructors can add or remove other instructors from exercises or definitions if they have access to these resources  
  - by creating an exercise or uploading an exercise definition, the instructor is automatically granted access to it - access can be removed by the other  instructor assigned to the exercise or definition  
  - instructor of the given exercise can see all teams in the exercise  
  - can manipulate with the exercises (start, stop, create, or remove)  
  - can add/remove trainees to/from teams of an exercise  
  - inherits all trainee permissions  
- **TRAINEE**  
  - can see only exercises where he/she was assigned  
  - can access only the data of the team to which he or she was assigned (cannot see the data of other teams for the same exercise)  
  - can use tools in exercise (sending emails, using tools,...)

### Process of Authorization
Every endpoint resolver (in REST API and GraphQL), which allows access to data that should not be visible to everyone, should be decorated by decorators. The INJECT authorization schema functions on the combinations of these decorators:

#### protected
  - takes as an argument permission needed for accessing the endpoint
  - Role-based access control -> checks whether the requester is assigned to a group with the required permission
  - if request is unauthenticated, it automatically denies access

```python
@protected(required_permission: str)
def resolver_function():
    pass
```
#### extra_protected
  - takes `Enum Check` value
  ```python
  class Check(str, Enum):
    TEAM_ID = "team_id"
    EXERCISE_ID = "exercise_id"
    DEFINITION_ID = "definition_id"
    LOG_ID = "log_id"
    THREAD_ID = "thread_id"
    VISIBLE_ONLY = "visible_only"
  ```
  - based on the value of the argument, checks whether the requester has access to a specific resource
  - utilizes Access-controll list
  - given argument must occur as a key-word argument of the endpoint resolver
  - works by extracting the Check keyword value and executing the relevant check function
  - if request is unauthenticated, it automatically denies access

```python
@extra_protected(check: Check)
def resolver_function(argument):
    pass
```
  - example usage:
```python
@extra_protected(Check.TEAM_ID)
def resolver_function(team_id: str):
    pass
```

#### input_object_protected
  - works on the same principle as **extra_protected** but instead of _Check enum argument, it takes the name of the input object argument.
  - internally uses the same check functions as **extra_protected**, but differs in the extraction of the needed value for the check.
```python
@input_object_protected(object_name: str)
def resolver():
    pass
```
- example usage:
```python
@input_object_protected("create_exercise_input")
def resolver_function(create_exercise_input: CreateExerciseInput):
    pass
```

The best authorization control and functionality are achieved by combining the mentioned decorators. This way, you can set up more granular and specific checks. For example:

```python
# Accessible only to the trainees assigned to the team with an ID equal to the team_id (inheritably for the instructors of the exercise, where the team with "team_id" belongs to)
@protected(Perms.view_trainee_info) # view_trainee_info is permission for the trainee role
@extra_protected(Checks.TEAM_ID)
def resolver(team_id: str):
    pass

# Accessible only to the instructors of the exercise, where the thread with "thread_id" was created
@protected(Perms.analytics_view) # analytics_view is the permission of the instructor role
@extra_protected(Check.THREAD_ID)
def resolver(thread_id):
    pass
```

### Additional notes
- Users with the ADMIN role can be added to the access control lists to be shown in exercises (if the admin acts as an instructor for some reason), but it is not necessary because ADMIN has control over every resource on the platform, whether he is assigned to it or not

## User onboarding
Users can be added to the platform via a .csv file in the following format:
```
username,group,tags,first_name,last_name
```
- **username** (mandatory)  
  - has to be a valid email address of the user (he will receive credentials via this email)  
- **group** (optional, implicitly "trainee")  
  - authorization role of the created user
  - values: trainee, instructor, or admin (shorts "t", "i" or "a" can be used as well) case is ignored  
Instructors cannot create users with higher privileges. admin (admin can be created only by admin users)  
- **tags** (optional)  
  - you can mark the newly created user by tags to make your work with assigning users to teams or exercises more convenient
  - format of the field: `tag1|tag2|tag3` (values separated by "|")  
- **first_name** (optional)  
  - first name of the created user if you want to identify them later  
- **last_name** (optional)  
  - last name of the created user if you want to identify them later  

As a separator for the column, you can use either `,` or `;`.