## Deployment using Docker/Nginx

INJECT Exercise Platform provides a simplified deployment method via Docker Compose. 
This method simplifies the deployment by creating necessary settings and pairings between the frontend and backend.
It also creates a singular endpoint for accessing INJECT Exercise Platform.

### Docker Setup

This solution consists of 5 containers: frontend, backend, redict, database and reverse proxy server.

Frontend and Backend are set up to communicate under a single endpoint handled by the reverse-proxy server (Nginx). 
PostgresSQL database handles management of user/exercise data.
Redict is used to allow multiple workers on the backend.
User-uploaded files, images and definition relevant content is stored on the Backend.
The reverse-proxy forwards requests either to Frontend or Backend.

Deployment is set up with persistent storage under two Docker Compose volumes called `inject-data` and `inject-pgdata`.
This volume contains any data generated by INJECT and current configuration of the platform.
At current state the platform can be effectively backed up assuming you back both volumes and `INJECT_SECRET_KEY` environment variable.

For HTTPS deployment it's needed to have HTTPS certificates.
At current state, there is a prepared configuration and script for deployment via Let's Encrypt certificates.
This script necessitates downtime of IXP to fully refresh or instantiate new SSL certificates.

### Environment Variables

All the technical configuration is done with environment variables.
These are prefixed under `INJECT_`.
Some additional variables are included, however these are inferred from the `INJECT_` prefixed environment variables,
which means that these do not have to be changed manually.

#### General Configuration

```
# sets up an appropriate hostname for the Nginx server
INJECT_DOMAIN=inject.localhost

# Secret key for cryptographic signing
INJECT_SECRET_KEY=your-at-least-fifty-or-more-characters-long-secret-key
```


- INJECT_DOMAIN: Defines the hostname for the Nginx server. This should be set to the **domain name** that points to your server's IP address.
- INJECT_SECRET_KEY: Used to provide cryptographic signing. Must be at least 50 characters long. Can be anything.
- INJECT_HOMEPAGE: The URL (link) of the deployed instance fo the INJECT platform. 
  The following URL is an example: _https://my-inject-domain.com_.
  This link is used in the welcome email.
- INJECT_MAX_UPLOAD_SIZE: Specifies the maximum body size of requests, including definition and file uploads. 
  The unit of this variable is bytes.
  The default value is set to 10MB (10485760 bytes).

#### Authentication, Authorization and Identification (AAI)

AAI modules ensures user authentication and sending of user credentials via email invitations. 
The emails are sent out using an external SMTP server.

The settings for sending emails is configured via the following environment variables. 

```
# uncomment and set appropriate values email to enable sending of AAI mails
# INJECT_EMAIL_HOST=
# INJECT_EMAIL_PORT=
# INJECT_EMAIL_HOST_USER=
# INJECT_EMAIL_HOST_PASSWORD=
# INJECT_EMAIL_SENDER_ADDRESS=
# INJECT_EMAIL_PROTOCOL=
```


- INJECT_EMAIL_HOST: SMTP server address for sending emails.
- INJECT_EMAIL_PORT: Port for the SMTP server. Port must be always set.
- INJECT_EMAIL_HOST_USER: Username for the SMTP server.
- INJECT_EMAIL_HOST_PASSWORD: Password for the SMTP server.
- INJECT_EMAIL_SENDER_ADDRESS: Email address used as the sender for outgoing emails.
- INJECT_EMAIL_PROTOCOL: Preferred protocol for communication with SMTP server.
  The choices (values) can be _ssl_ or _tls_ (case-insensitive).
  This variable can also be left undefined - no encryption will be used.
  Please note, that setting of `INJECT_EMAIL_PROTOCOL` does not automatically set the port and port must be always set.
- INJECT_EMAIL_TIMEOUT: Specifies a timeout in seconds for blocking operations tied to emails like the connection attempt to SMTP.
  The default value is 10 seconds.

#### Logging and Debugging

The backend logs various actions and possible errors in a special log file.
The location of the logfile can be changed by setting the `INJECT_LOGS` variable to a new path to a _file_.
This path is relative from the `/backend` directory in the Docker container.

```
# enable logging to a specific file
# INJECT_LOGS=data/backend-logs.log
```

- INJECT_LOGS: Path to the log file. Set to `backend-logs.log` by default.

#### Host and Connection Settings

These parameters are set up by default in the `.env` file, it's advised to not modify them,
unless you want to specifically modify how IXP interacts with Frontend and Backend.
Individual Docker compose setups these settings setup accordingly to the set level of communication. 

```
# host parameters that should be set up for the client and server to interact correctly
INJECT_HOST_ADDRESSES=$INJECT_DOMAIN
VITE_HTTPS_HOST=$INJECT_DOMAIN
CORS_ALLOWED_ORIGINS=https://$INJECT_DOMAIN
```

- INJECT_HOST_ADDRESSES: Comma-separated list of addresses where the backend can be reached.
  Misconfiguration may lead to an inaccessible Backend instance.
- VITE_HTTPS_HOST: Defines the HTTP/S host for which the frontend application should connect to,
  the hostname should point to an instance of Backend application.
- CORS_ALLOWED_ORIGINS: Comma-separated list of origins allowed for Cross-Origin Resource Sharing (CORS).
  Example value: `https://inject.muni.cz,https://alternative-inject.muni.cz`. 
  Implicitly set-up values by backend include: `http://localhost:5173`, `http://127.0.0.1:5173`, `http://localhost:8000`, `http://127.0.0.1:8000`.
  Misconfiguration may lead to issues stemming from CORS policies setup in the web browser.

#### Database settings

```
INJECT_DB_PASS=a-reasonably-secure-and-long-password
INJECT_DB_USER=inject
```

- INJECT_DB_PASS: PostgresSQL user password for which the IXP Backend has to log in to
- INJECT_DB_USER: PostgresSQL username for which the IXP Backend has to log in to

#### Redict settings

```
INJECT_REDICT_HOST=inject-redict
```

- INJECT_REDICT_HOST: Host address where Redict is served.

#### Backend settings

```
INJECT_WORKER_COUNT=4
```

- INJECT_WORKER_COUNT: Number of workers to use for deployment.
  If unspecified, uses only 1 worker.
  It is recommended to use `2 x CPU_COUNT` number of workers.

## Good practices

At current state it's advised:
- to back up data located in `inject-data` and `inject-pgdata` volume
- to back up `INJECT_SECRET_KEY`, this key is critical to ensure proper functioning of AAI,
  if the same database is used with a different key, users will not be able to log in

<div class="navigation" markdown>
  [&larr; Installation overview](./overview.md){ .md-button }
</div>
