---
tags: used_in_docs
---
## Deployment using Nginx

INJECT Exercise Platform provides a simplified deployment method via Docker Compose. 
This method simplifies the deployment by creating necessary settings and pairings between the frontend and backend and also creating a singular endpoint for accessing INJECT Exercise Platform.

This solution consists of 3+1 containers: frontend, backend, Nginx Proxy (as a front-facing router) and [Certbot](https://certbot.eff.org/).

Frontend and Backend are setup to communicate under a single endpoint handled by Nginx Proxy. The Proxy forwards appropriate requests either to Frontend or Backend. 
Certbot handles the generation of HTTPS certificates if needed. 
Currently it only simplifies the creation of new certificates, but it is not automated.

Deployment is setup with persistent storage under a Docker Compose volume called `backend-data`. This volume contains any data generated by INJECT. At current state it's adviced if there is any manipulation to be done on said containers (ex. restarting, shutting them down, updating, etc.) to ensure that no exercises are running at given point of time.

## Environment variables

For environment variables, please refer to the `.env` file. 
The example file contains an ideal set of environment variables to set up the platform. 
Below, the variables are categorized and described for better understanding.

### General Configuration
```
# sets up an appropriate hostname for the Nginx server
INJECT_DOMAIN=inject.localhost

# Secret key for cryptographic signing
INJECT_SECRET_KEY=your-at-least-fifty-or-more-characters-long-secret-key
```


- INJECT_DOMAIN: Defines the hostname for the Nginx server. This should be set to the **domain name** that points to your server's IP address.
- INJECT_SECRET_KEY: Used to provide cryptographic signing. Must be at least 50 characters long. Can be anything.


### Authentication, Authorization and Identification (AAI)

AAI enables user authentication and sending user credentials via email. 
The emails are sent out using an external SMTP server.

The settings for sending emails is configured the following environment variables in `.env` file. 

```
# for email sending of AAI mails
INJECT_EMAIL_HOST=
INJECT_EMAIL_PORT=
INJECT_EMAIL_HOST_USER=
INJECT_EMAIL_HOST_PASSWORD=
INJECT_EMAIL_SENDER_ADDRESS=
INJECT_EMAIL_PROTOCOL=
```

- INJECT_NOAUTH: Comment this out to to enable AAI (enabled by default).

#### Email Configuration for AAI

- INJECT_EMAIL_HOST: SMTP server address for sending emails.
- INJECT_EMAIL_PORT: Port for the SMTP server.
- INJECT_EMAIL_HOST_USER: Username for the SMTP server.
- INJECT_EMAIL_HOST_PASSWORD: Password for the SMTP server.
- INJECT_EMAIL_SENDER_ADDRESS: Email address used as the sender for outgoing emails.
- INJECT_EMAIL_PROTOCOL: Preferred protocol for communication with SMTP server. The choices (values) can be _ssl_ or _tls_ (case insensitive). This variable can also be left undefined - no encryption will be used.

## HTTPS Deployment

For HTTPS deployment, you need to generate HTTPS certificates.

Use the `generate-cert.sh` script, which calls [Certbot](https://certbot.eff.org/) and outputs certificates in the `./certbot` directory of the Compose script for the given `INJECT_DOMAIN` setup in the `.env` file.  Before full deployment ensure that `.env` file has correctly set protocols for HTTP communication. That is - `https` and `wss`, failing to do so may lead to connection issues. This applies for `CORS_ALLOWED_ORIGINS` and `VITE_HTTP_WS`
For more information on configuration of this script, please refer to [CertBot Configuration](#certbot-configuration)

When the certificates are generated, execute `deploy-with-https.sh` which will run INJECT Exercise Platform. 
This script launches three Docker Compose layers that form a complete script.
The first one is the basal binding compose file (`docker-compose.yml`), which establishes the basic bindings between frontend, backend, and Nginx Proxy.
The second one (`compose-from-registry.yml`) sets up appropriate import paths for frontend and backend images.
The last one (`compose-with-nginx-https.yml`) sets up Nginx Proxy to run in HTTPS mode under ports `80` and `443` and generates the necessary nginx configuration from `nginx-https.conf.template`.

## HTTP Deployment

Although HTTPS is the recommended and default deployment method, you can still deploy the platform using HTTP. Before full deployment ensure that `.env` file has correctly set protocols for HTTP communication. That is - `http` and `ws`, failing to do so may lead to connection issues. This applies for `CORS_ALLOWED_ORIGINS` and `VITE_HTTP_WS`

For HTTP deployment, please refer to the `deploy.sh` script in the root of the folder.
The script is similar to `deploy-with-https.sh` with one caveat, and that's the usage of `compose-with-nginx.yml` instead of `compose-with-nginx-https.yml` which by essence does the same but uses a different NGINX template (`nginx.conf.template`) and runs on port `80`.

## Creating an Admin

After deployment, you might need to create a admin to manage the platform. Follow these steps in order to create a admin:


 - List the running Docker containers to find the **NAME** of the backend container:
    ```
    docker ps
    ```
 - Execute a shell session in the backend container. Replace `CONTAINERNAME` with the actual container **NAME** of the backend service:
    ```
    docker exec -it CONTAINERNAME python manage.py shell
    ```

- Create a admin using the Django shell. Replace `email@test.com` and `password` with your desired admin credentials.:

    ``` python
    from user.models import User
    User.objects.create_superuser(username="email@test.com", password="password")
    ```

## Optional and additional settings

### Disabling AAI

To disable AAI, modify the `.env` file and uncomment the `INJECT_NOAUTH` variable. Doing so disables the AAI.

### CertBot Configuration

If you want to use HTTPS, you can generate certificates using CertBot. Configure the following environment variable in your `.env` file:

```
# for certbot generation, use with certbot-generator.yml
CERTBOT_EMAIL=noemail@inject.localhost
```

- CERTBOT_EMAIL: Email address used for generating HTTPS certificates with CertBot.

### Logging and Debugging

If you need to enable logging or debugging for the backend, configure the following environment variables in your .env file:

```
# enable logging to a specific file
# INJECT_LOGS=data/

# enable debug mode for the backend
INJECT_DEBUG=
```

- INJECT_LOGS: Path to the log file. Uncomment and set this variable to enable logging.
- INJECT_DEBUG: If set, enables debug mode for the backend.

### Host and Connection Settings
```
# host parameters that should be set up for the client and server to interact correctly,
# do not touch unless you know what you are doing
INJECT_HOST_ADDRESSES=$INJECT_DOMAIN
VITE_HTTP_HOST=$INJECT_DOMAIN
VITE_HTTP_WS=ws://$INJECT_DOMAIN/ttxbackend/api/v1/subscription
CORS_ALLOWED_ORIGINS=http://$INJECT_DOMAIN
```

- INJECT_HOST_ADDRESSES: Specifies the address where the backend can be reached.
- VITE_HTTP_HOST: Defines the HTTP host for the frontend.
- VITE_HTTP_WS: Sets up the WebSocket address for the frontend.
- CORS_ALLOWED_ORIGINS: Lists the origins allowed for Cross-Origin Resource Sharing (CORS). Implicitly set-up values by backend include: `http://localhost:5173`, `http://127.0.0.1:5173`, `http://localhost:8000`, `http://127.0.0.1:8000`.

## Troubleshooting

If you encounter issues during deployment, consider the following steps:

- **Executable Permissions**: Ensure that the deployment scripts have executable permissions. Run the following command to set the appropriate permissions:
    ```
    chmod +x deploy.sh 01-substitute-env.sh
    chmod +x deploy-with-https.sh generate-cert.sh  # If using HTTPS deployment
    ```

- **INJECT_DOMAIN Setting**: Verify that the INJECT_DOMAIN environment variable is set to a domain name and not a static IP address. The platform requires a domain name to function correctly. For local test deployments please use reserved TLD `.localhost` (example: `inject.localhost`)

- **Frontend's error log in Developer Console is reporting mixed-origin connection issues**: Verify that connection settings in the environment varibles are setup correctely. If using HTTP deployment, that they use `http://` and `ws://` protocol, and when using HTTPS, then `https://` and `wss://`. After fixing the issues please reset your cookies and local storage.
