## Deployment using Docker/Nginx

INJECT Exercise Platform provides a simplified deployment method via Docker Compose.
This method simplifies the deployment by creating necessary settings and pairings between the frontend and backend.
It also creates a singular endpoint for accessing IXP.

### Docker Setup

This solution consists of 5 containers: frontend, backend, redict, database and reverse proxy server.

Frontend and Backend are set up to communicate under a single endpoint handled by the reverse-proxy server (Nginx).
PostgreSQL database handles the management of user/exercise data.
Redict is used to allow multiple workers on the backend.
User-uploaded files, images and definition-related content are stored on the Backend.
The reverse proxy forwards requests either to Frontend or Backend.

Deployment is set up with persistent storage under two Docker Compose volumes called `inject-data` and `inject-pgdata`.
This volume contains any data generated by IXP and the current configuration of the platform.
At the current state, the platform can be effectively backed up, assuming you back up both volumes and the `INJECT_SECRET_KEY` environment variable.

For HTTPS deployment, HTTPS certificates are needed.
At the current state, there is a prepared configuration and script for deployment via Let's Encrypt certificates.
This script necessitates downtime of IXP to fully refresh or instantiate new SSL certificates.

### Environment Variables

All the technical configuration is done with environment variables.
These are prefixed under `INJECT_`.
Some additional variables are included, however these are inferred from the `INJECT_` prefixed environment variables,
which means that these do not have to be changed manually.

The available variables are:

- `INJECT_DOMAIN` - Domain where your instance of IXP is available.
- `INJECT_SECRET_KEY` - Used to provide cryptographic signing. Must be at least 50 long characters string.
- `INJECT_HOST_ADDRESSES` - A comma-separated list of allowed hosts.
- `VITE_HTTPS_HOST` - Defines the HTTPS host for which the frontend application should connect to, the hostname should point to an instance of Backend application.
- `CORS_ALLOWED_ORIGINS` - A comma-separated list of allowed origins for Cross-Origin Resource Sharing (CORS).
- `INJECT_EMAIL_HOST` - SMTP server address.
- `INJECT_EMAIL_PORT` - Port to use for the SMTP server defined in INJECT_EMAIL_HOST.
- `INJECT_EMAIL_HOST_USER` - Username to use for authentication to the SMTP server defined in INJECT_EMAIL_HOST.
- `INJECT_EMAIL_HOST_PASSWORD` - Password to use for the SMTP server defined in INJECT_EMAIL_HOST. This setting is used in conjunction with INJECT_EMAIL_HOST_USER when authenticating to the SMTP server.
- `INJECT_EMAIL_SENDER_ADDRESS` - The sender address for automatic emails.
- `INJECT_EMAIL_PROTOCOL` - Optional variable for defining the preferred protocol for communication with SMTP server. The choices (values) can be ssl or tls (case insensitive). This variable can also be left undefined - no encryption will be used.
- `INJECT_HOMEPAGE` - The URL of the homepage.
- `INJECT_LOGS` - Path to a file where to save logs.
- `INJECT_DB_HOST` - Host address of the database server.
- `INJECT_DB_USER` - Username for the database.
- `INJECT_DB_PASS` - Password for the database.
- `INJECT_REDICT_HOST` - If set, enables Redict. Host address where Redict is running.
- `INJECT_WORKER_COUNT` - Number of worker processes to run. This can impact performance and resource utilization.

You can use the default `.env` file to set up your configuration:

```
# sets up appropriate hostname for the Nginx server
INJECT_DOMAIN=inject.localhost

# Secret key for cryptographic signing
INJECT_SECRET_KEY=your-at-least-fifty-or-more-characters-long-secret-key

# host parameters that should be setup for client and server to interact correctly,
# do not touch unless you know what you are doing
INJECT_HOST_ADDRESSES=$INJECT_DOMAIN
VITE_HTTPS_HOST=$INJECT_DOMAIN
CORS_ALLOWED_ORIGINS=https://$INJECT_DOMAIN

# uncomment and set appropriate values email to enable sending of AAI mails
# INJECT_EMAIL_HOST=
# INJECT_EMAIL_PORT=
# INJECT_EMAIL_HOST_USER=
# INJECT_EMAIL_HOST_PASSWORD=
# INJECT_EMAIL_SENDER_ADDRESS=
# INJECT_EMAIL_PROTOCOL=
INJECT_HOMEPAGE=https://${INJECT_DOMAIN}/

# redirect logs to the `data` directory
INJECT_LOGS=data/backend-logs.log

# inject postgres DB host, this HOST is preset to hostname as set in compose.yml
INJECT_DB_HOST=inject-postgres

# database authentication settings, please setup appropriate and secure password in INJECT_DB_PASS
INJECT_DB_USER=inject
INJECT_DB_PASS=a-reasonably-secure-and-long-password

# redict settings, used from backend v2.4.0
INJECT_REDICT_HOST=inject-redict

# set the number of workers for backend
# it is recommended to use 2 x CPU_COUNT number of workers
# used from backend v2.5.0
INJECT_WORKER_COUNT=4
```

## Backup

It's advised to back up:

- data located in the `inject-data` and the `inject-pgdata` docker volume,
- and the `INJECT_SECRET_KEY` variable. This key is critical to ensure proper functioning of AAI:
  if the same database is used with a different key, users will not be able to log in.

<div class="navigation" markdown>
  [&larr; Installation overview](./overview.md){ .md-button }
</div>
