// TODO: don't rely on in-place filtering, use cache

import { NonIdealState } from '@blueprintjs/core'
import { css } from '@emotion/css'
import type {
  ActionLogFlat,
  SimplifiedActionLog,
} from '@inject/graphql/fragment-types'
import { useVirtualizer } from '@tanstack/react-virtual'
import type { FC } from 'react'
import { useMemo, useRef } from 'react'
import LogItem from './LogItem'
import { selectTodoCat } from './utils'

const InstructorTodoLog: FC<
  {
    height: string
    done?: boolean
    noAttention?: boolean
    selectedTeams: string[]
  } & (
    | {
        actionLogs: SimplifiedActionLog[]
        actionLogIds: null
      }
    | {
        actionLogs: null
        actionLogIds: ActionLogFlat[]
      }
  )
> = ({
  actionLogs,
  actionLogIds,
  done,
  height,
  selectedTeams,
  noAttention,
}) => {
  const filtered = useMemo<SimplifiedActionLog[] | ActionLogFlat[]>(
    () =>
      (actionLogs || actionLogIds)
        .filter(actionLog => noAttention || actionLog.requiresAttention)
        .filter(actionLog => selectTodoCat(actionLog) === done)
        .filter(actionLog => selectedTeams.includes(actionLog.team.id)),
    [actionLogs, actionLogIds, noAttention, done, selectedTeams]
  )

  // The scrollable element for your list
  const parentRef = useRef<HTMLDivElement>(null)
  // The virtualizer
  const rowVirtualizer = useVirtualizer({
    count: filtered.length,
    getScrollElement: () => parentRef.current,
    estimateSize: () => 120,
    overscan: 5,
  })
  const items = rowVirtualizer.getVirtualItems()

  if (filtered.length === 0) {
    return (
      <NonIdealState
        className={css`
          height: ${height};
          width: 100%;
        `}
        icon='low-voltage-pole'
        title='No inject items'
        description='There are no log items that can be shown in this category. Please await for new injects'
      />
    )
  }

  return (
    <div
      ref={parentRef}
      style={{
        marginTop: '0.25rem',
        overflow: 'auto',
        width: '100%',
        height,
      }}
    >
      <div
        style={{
          height: `${rowVirtualizer.getTotalSize()}px`,
          width: '100%',
          position: 'relative',
        }}
      >
        <div
          style={{
            position: 'absolute',
            top: 0,
            left: 0,
            width: '100%',
            transform: `translateY(${items[0]?.start ?? 0}px)`,
          }}
        >
          {items.map(item => (
            <div
              style={{
                padding: '0.5rem 0',
              }}
              key={item.key}
              data-index={item.index}
              ref={rowVirtualizer.measureElement}
            >
              <LogItem
                actionLog={
                  actionLogs
                    ? (filtered[item.index] as SimplifiedActionLog)
                    : null
                }
                actionLogId={filtered[item.index].id}
              />
            </div>
          ))}
        </div>
      </div>
    </div>
  )
}

export default InstructorTodoLog
