import type { Path } from '@/router'
import { useNavigate } from '@/router'
import { Button, NonIdealState } from '@blueprintjs/core'
import { css } from '@emotion/css'
import type { FC, ReactNode } from 'react'
import { memo } from 'react'
import FaqSection from '../FaqSection'
import { GENERIC_CONTENT } from '../assets/generalContent'
import { PAGE_INFORMATION } from '../assets/pageInformation'
import type { PageNames } from '../types'

const editorPage = css`
  display: grid;
  grid-template-rows: auto auto 1fr auto;
  height: 100vh;
  padding: 0rem 1rem;
`

const header = css`
  display: flex;
  justify-content: space-between;
  align-items: center;
`

interface EditorPageProps {
  pageKey: PageNames
  children: ReactNode
  headerElement?: ReactNode
  descriptionElement?: ReactNode
  prevPath: Path
  nextPath?: Path
  nextDisabled?: boolean
  nextVisible?: boolean
  pageVisible?: boolean
}

const EditorPage: FC<EditorPageProps> = ({
  pageKey,
  children,
  headerElement,
  descriptionElement,
  prevPath,
  nextPath,
  nextDisabled,
  nextVisible,
  pageVisible,
}) => {
  const nav = useNavigate()
  const content = PAGE_INFORMATION[pageKey]

  if (!pageVisible) {
    return <NonIdealState title='Page not found' icon='warning-sign' />
  }

  return (
    <div className={editorPage}>
      <div className={header}>
        <h1>{content.title}</h1>
        {headerElement}
      </div>
      <div>
        <p
          dangerouslySetInnerHTML={{
            __html: content.description,
          }}
          style={{ marginBottom: '1rem' }}
        />
        {descriptionElement}
      </div>
      <div style={{ overflowY: 'auto' }}>
        {children}
        {content.faq && content.faq.length > 0 && (
          <FaqSection questions={content.faq} />
        )}
      </div>
      <div
        style={{
          display: 'flex',
          justifyContent: 'center',
          padding: '0.5rem 0',
        }}
      >
        <Button
          type='button'
          onClick={() => nav(prevPath)}
          text={GENERIC_CONTENT.buttons.back}
          icon='arrow-left'
          style={{
            marginRight: '0.5rem',
          }}
        />
        {(nextVisible === undefined || nextVisible) && (
          <Button
            type='button'
            onClick={() => nav(nextPath || '/')}
            text={GENERIC_CONTENT.buttons.next}
            intent='primary'
            rightIcon='arrow-right'
            disabled={nextDisabled}
          />
        )}
      </div>
    </div>
  )
}

export default memo(EditorPage)
