import { formatTimestamp } from '@/analyst/utilities'
import Label from '@/users/UsersSelection/Label'
import type { Intent } from '@blueprintjs/core'
import { Button, Colors, Dialog, DialogBody } from '@blueprintjs/core'
import { Cross, Tick } from '@blueprintjs/icons'
import { css } from '@emotion/css'
import type { Exercise } from '@inject/graphql/fragment-types'
import { dialog, dialogBody } from '@inject/shared/css/dialog'
import { breakWord } from '@inject/shared/css/textOverflow'
import { useState, type FC } from 'react'

const table = css`
  table-layout: fixed;
  width: 100%;
  text-align: left;

  th,
  td {
    padding: 0.5rem;
    vertical-align: top;
  }

  th {
    width: 11ch;
  }
  td {
    ${breakWord}
  }
`

interface InfoButtonProps {
  exercise: Exercise
  intent?: Intent
}

const InfoButton: FC<InfoButtonProps> = ({ exercise, intent }) => {
  const [dialogOpen, setDialogOpen] = useState(false)

  const {
    name,
    definition,
    teams,
    exerciseStart,
    finished,
    running,
    createdBy,
    timestampCreated,
  } = exercise

  return (
    <>
      <Button
        icon='info-sign'
        title='Exercise information'
        active={dialogOpen}
        onClick={() => setDialogOpen(true)}
        intent={intent}
      />
      <Dialog
        isOpen={dialogOpen}
        onClose={() => setDialogOpen(false)}
        title={<span title={`Exercise ${name}`}>{`Exercise ${name}`}</span>}
        className={dialog}
      >
        <DialogBody className={dialogBody}>
          <table className={table}>
            <tbody>
              <tr>
                <th>Name:</th>
                <td title={name}>{name}</td>
              </tr>
              <tr>
                <th>Definition:</th>
                <td>{definition?.name || definition?.id}</td>
              </tr>
              <tr>
                <th>Teams:</th>
                <td>{teams.length}</td>
              </tr>
              <tr>
                <th>Started at:</th>
                <td>
                  {exerciseStart ? (
                    formatTimestamp(exerciseStart)
                  ) : (
                    <Cross color={Colors.RED3} />
                  )}
                </td>
              </tr>
              <tr>
                <th>Running:</th>
                <td>
                  {running ? (
                    <Tick color={Colors.GREEN3} />
                  ) : (
                    <Cross color={Colors.RED3} />
                  )}
                </td>
              </tr>
              <tr>
                <th>Finished:</th>
                <td>
                  {finished ? (
                    <Tick color={Colors.GREEN3} />
                  ) : (
                    <Cross color={Colors.RED3} />
                  )}
                </td>
              </tr>
              <tr>
                <th>Created by:</th>
                <td>
                  {createdBy ? <Label user={createdBy} /> : 'a deleted user'}
                </td>
              </tr>
              <tr>
                <th>Created at:</th>
                <td>{formatTimestamp(timestampCreated)}</td>
              </tr>
            </tbody>
          </table>
        </DialogBody>
      </Dialog>
    </>
  )
}

export default InfoButton
