import type { LinkType } from '@/components/LinkButton'
import type { NonIdealStateProps } from '@blueprintjs/core'
import { NonIdealState } from '@blueprintjs/core'
import { css } from '@emotion/css'
import { useTypedQuery } from '@inject/graphql/graphql'
import { GetTeamChannelLogsFlat } from '@inject/graphql/queries'
import notEmpty from '@inject/shared/utils/notEmpty'
import { type FC, memo, useMemo } from 'react'
import ListRenderer from './ListRenderer'

interface ActionLogProps {
  teamId: string
  exerciseId: string
  channelId: string
  inInstructor: boolean
  getFullscreenLink: (actionLogId: string) => LinkType
  noDataProps?: NonIdealStateProps
  scrollToLast?: boolean
}

/**
 * The component expects that the subscription is active for the given teamId
 */
const ActionLog: FC<ActionLogProps> = ({
  inInstructor,
  teamId,
  exerciseId,
  channelId,
  getFullscreenLink,
  noDataProps,
  scrollToLast,
}) => {
  const [{ data }] = useTypedQuery({
    query: GetTeamChannelLogsFlat,
    variables: {
      teamId,
      channelId,
    },
    context: useMemo(
      () => ({
        suspense: true,
      }),
      []
    ),
  })
  const actionLogs = (data?.teamChannelLogs || [])?.filter(notEmpty)

  if (actionLogs.length === 0) {
    return (
      <NonIdealState
        icon='low-voltage-pole'
        title='No injects found'
        description='There are no injects in this channel yet'
        className={css`
          height: 80vh;
          width: 100%;
        `}
        {...noDataProps}
      />
    )
  }

  return (
    <ListRenderer
      actionLogs={actionLogs}
      scrollToLast={scrollToLast}
      channelId={channelId}
      exerciseId={exerciseId}
      getFullscreenLink={getFullscreenLink}
      inInstructor={inInstructor}
      teamId={teamId}
    />
  )
}

export default memo(ActionLog)
