import { EmailSelection } from '@/analyst/utilities'
import type { LinkType } from '@/components/LinkButton'
import LinkButton from '@/components/LinkButton'
import {
  OPEN_COMPOSE_EVENT_TYPE,
  OPEN_REPLY_EVENT_TYPE,
} from '@/email/EmailFormOverlay'
import { useNavigate } from '@/router'
import { Button, ButtonGroup, Divider } from '@blueprintjs/core'
import { css, cx } from '@emotion/css'
import type { EmailThread } from '@inject/graphql/fragments/EmailThread.generated'
import { type FC } from 'react'
import DraftLogCards from './DraftLogCards'
import ThreadLogCards from './ThreadLogCards'

const threadLog = css`
  display: flex;
  flex-shrink: 10;
  flex-direction: column;
  overflow: auto;
  flex: 1;
`

const header = css`
  padding: 0.5rem 1rem 0;
  display: flex;
  flex-wrap: wrap;
  justify-content: space-between;
`

const buttons = (inInstructor?: boolean) => css`
  flex-wrap: wrap;

  @media (max-width: ${inInstructor ? '52rem' : '42rem'}) {
    flex-direction: column;
  }
`
const buttonsVertical = (inInstructor?: boolean) => css`
  @media (max-width: ${inInstructor ? '130rem' : '100rem'}) {
    flex-direction: column;
  }
`

interface ThreadLogProps {
  teamId: string
  emailThreads: EmailThread[]
  selectedTab: EmailSelection
  receivedLink: LinkType
  sentLink: LinkType
  draftsLink?: LinkType
  selectedEmailThreadId?: string
  onClick: (emailThread: EmailThread) => void
  inInstructor?: boolean
  inAnalyst?: boolean
  allowComposing?: boolean
  composeButtonTitle?: string
  onCompose?: () => void
}

const ThreadLog: FC<ThreadLogProps> = ({
  teamId,
  emailThreads,
  selectedTab,
  receivedLink,
  sentLink,
  inInstructor,
  draftsLink,
  selectedEmailThreadId,
  onClick,
  inAnalyst = false,
  allowComposing = false,
  composeButtonTitle,
  onCompose,
}) => {
  const nav = useNavigate()

  return (
    <div className={threadLog}>
      <div className={header}>
        <ButtonGroup
          minimal
          className={cx({
            [buttons(inInstructor)]: true,
            [buttonsVertical(inInstructor)]:
              selectedEmailThreadId !== undefined,
          })}
        >
          <LinkButton
            link={receivedLink}
            button={{
              text: `Received${inInstructor ? ' by team' : ''}`,
              icon: 'inbox',
              active: selectedTab === EmailSelection.RECEIVED,
            }}
          />
          <Divider />
          <LinkButton
            link={sentLink}
            button={{
              text: `Sent${inInstructor ? ' by team' : ''}`,
              icon: 'send-message',
              active: selectedTab === EmailSelection.SENT,
            }}
          />
          {draftsLink && (
            <>
              <Divider />
              <LinkButton
                link={draftsLink}
                button={{
                  text: 'Drafts',
                  icon: 'document',
                  active: selectedTab === EmailSelection.DRAFTS,
                }}
              />
            </>
          )}
        </ButtonGroup>
        <div>
          <Button
            minimal
            intent='primary'
            text='Compose'
            icon='edit'
            onClick={onCompose}
            disabled={!allowComposing}
            title={composeButtonTitle}
          />
        </div>
      </div>

      <Divider />

      {selectedTab === EmailSelection.DRAFTS ? (
        <DraftLogCards
          onClick={emailThread => {
            if (emailThread) {
              onClick(emailThread)
              window.dispatchEvent(
                new CustomEvent(OPEN_REPLY_EVENT_TYPE, {
                  detail: {
                    emailThread,
                  },
                })
              )
            } else {
              if (!draftsLink) return
              // eslint-disable-next-line prefer-spread
              nav.apply(null, draftsLink)
              window.dispatchEvent(new CustomEvent(OPEN_COMPOSE_EVENT_TYPE))
            }
          }}
          teamId={teamId}
          inInstructor={inInstructor}
          selectedEmailThreadId={selectedEmailThreadId}
        />
      ) : (
        <ThreadLogCards
          teamId={teamId}
          emailThreads={emailThreads}
          selectedTab={selectedTab}
          onClick={onClick}
          inAnalyst={inAnalyst}
          selectedEmailThreadId={selectedEmailThreadId}
        />
      )}
    </div>
  )
}

export default ThreadLog
