import { ellipsized } from '@/analyst/utilities'
import { Checkbox, Icon, IconSize } from '@blueprintjs/core'
import { css, cx } from '@emotion/css'
import type { Dispatch, SetStateAction } from 'react'
import { useCallback, type FC } from 'react'
import { checkbox, selectHeaderCell, sticky, tableCell } from './classes'
import type { Column } from './typing'
import { SortingFunction } from './typing'

const verticallyCentered = css`
  display: flex;
  align-items: center;
`

type HeadProps = {
  columns: Omit<Column, 'display'>[]
  defaultSortByColumn?: number
  defaultSortingFunction?: SortingFunction
  selectedColumnIndex: number
  setSelectedColumnIndex: Dispatch<SetStateAction<number>>
  sortingFunction: SortingFunction[]
  setSortingFunction: Dispatch<SetStateAction<SortingFunction[]>>
} & (
  | {
      allSelected?: never
      onSelectAll?: never
      onDeselectAll?: never
    }
  | {
      allSelected: boolean
      onSelectAll: () => void
      onDeselectAll: () => void
    }
)

const Head: FC<HeadProps> = ({
  columns,
  selectedColumnIndex,
  setSelectedColumnIndex,
  sortingFunction,
  setSortingFunction,
  allSelected,
  onSelectAll,
  onDeselectAll,
}) => {
  const handleClick = useCallback(
    (index: number) => {
      if (selectedColumnIndex === index) {
        setSortingFunction(prev => [
          ...prev.slice(0, index),
          prev[index] === SortingFunction.ASC
            ? SortingFunction.DESC
            : SortingFunction.ASC,
          ...prev.slice(index + 1, prev.length),
        ])
      } else {
        setSelectedColumnIndex(index)
      }
    },
    [selectedColumnIndex, setSelectedColumnIndex, setSortingFunction]
  )

  return (
    <thead>
      <tr>
        {onSelectAll && (
          <th className={cx(sticky, tableCell, selectHeaderCell)}>
            <Checkbox
              className={checkbox}
              checked={allSelected}
              onChange={allSelected ? onDeselectAll : onSelectAll}
            />
          </th>
        )}
        {columns.map((column, i) => (
          <th
            key={column.id}
            className={cx(sticky, tableCell, ellipsized)}
            style={column.style}
            title={column.name}
          >
            <div
              className={cx(ellipsized, verticallyCentered)}
              style={{ display: 'flex', justifyContent: 'space-between' }}
            >
              <div className={ellipsized} title={column.name}>
                {column.name}
              </div>
              <Icon
                icon={
                  sortingFunction[i] === SortingFunction.ASC
                    ? 'arrow-up'
                    : 'arrow-down'
                }
                color={
                  selectedColumnIndex === i
                    ? 'rgb(95, 107, 124)'
                    : 'rgba(95, 107, 124, 0.5)'
                }
                style={{ cursor: 'pointer' }}
                size={IconSize.LARGE}
                onClick={() => {
                  handleClick(i)
                }}
              />
            </div>
          </th>
        ))}
      </tr>
    </thead>
  )
}
export default Head
