import FileViewRedirectButton from '@/components/FileViewRedirectButton'
import type { ActionLog } from '@inject/graphql/fragments/ActionLog.generated'
import type { CustomInjectDetails } from '@inject/graphql/fragments/CustomInjectDetails.generated'
import type { EmailDetails } from '@inject/graphql/fragments/EmailDetails.generated'
import type { QuestionnaireDetails } from '@inject/graphql/fragments/QuestionnaireDetails.generated'
import type { ToolDetails } from '@inject/graphql/fragments/ToolDetails.generated'
import type { InjectDetailsType } from '@inject/graphql/types'
import type { FC } from 'react'
import { useContext } from 'react'
import ExerciseContext from '../ExerciseContext'
import useFormatTimestamp from '../useFormatTimestamp'
import { td, th } from './detailStyles'

const getContent = (actionLog: ActionLog, exerciseId: string) => {
  let details
  switch (actionLog.type) {
    case 'INJECT':
      details = actionLog.details as InjectDetailsType
      return (
        <>
          {details.inject.organization && (
            <tr>
              <th className={th}>Organization</th>
              <td className={td}>{details.inject.organization}</td>
            </tr>
          )}
          <tr>
            <th className={th}>Name</th>
            <td className={td}>{details.inject.name}</td>
          </tr>
          <tr>
            <th className={th}>Content</th>
            <td
              className={td}
              aria-label='renderedContent'
              dangerouslySetInnerHTML={{
                __html: details.content.rendered,
              }}
            />
          </tr>
          {details.content.fileInfo && (
            <tr>
              <th className={th}>Attached file</th>
              <td className={td}>
                <FileViewRedirectButton
                  fileInfo={details.content.fileInfo}
                  teamId={actionLog.team.id}
                  exerciseId={exerciseId}
                  inInstructor
                />
              </td>
            </tr>
          )}
        </>
      )
    case 'CUSTOM_INJECT':
      details = actionLog.details as CustomInjectDetails
      return (
        <>
          <tr>
            <th className={th}>Content</th>
            <td
              className={td}
              aria-label='renderedContent'
              dangerouslySetInnerHTML={{
                __html: details.content.rendered,
              }}
            />
          </tr>
          {details.content.fileInfo && (
            <tr>
              <th className={th}>Attached file</th>
              <td className={td}>
                <FileViewRedirectButton
                  fileInfo={details.content.fileInfo}
                  teamId={actionLog.team.id}
                  exerciseId={exerciseId}
                  inInstructor
                />
              </td>
            </tr>
          )}
        </>
      )
    case 'TOOL':
      details = actionLog.details as ToolDetails
      return (
        <>
          <tr>
            <th className={th}>Tool</th>
            <td className={td}>{details.tool.name}</td>
          </tr>
          <tr>
            <th className={th}>Argument</th>
            <td className={td}>{details.argument}</td>
          </tr>
          <tr>
            <th className={th}>Response</th>
            <td
              className={td}
              aria-label='renderedContent'
              dangerouslySetInnerHTML={{
                __html: details.content.rendered,
              }}
            />
          </tr>
          {details.content.fileInfo && (
            <tr>
              <th className={th}>Attached file</th>
              <td className={td}>
                <FileViewRedirectButton
                  fileInfo={details.content.fileInfo}
                  teamId={actionLog.team.id}
                  exerciseId={exerciseId}
                  inInstructor
                />
              </td>
            </tr>
          )}
        </>
      )
    case 'FORM':
      details = actionLog.details as QuestionnaireDetails
      return (
        <>
          <tr>
            <th className={th}>Title</th>
            <td className={td}>{details.title}</td>
          </tr>
          {/* TODO: */}
        </>
      )
    case 'EMAIL':
      details = actionLog.details as EmailDetails
      return (
        <>
          <tr>
            <th className={th}>Sender</th>
            <td className={td}>{details.sender.address}</td>
            <th className={th}>Subject</th>
            <td className={td}>{details.thread.subject}</td>
            <tr>
              <th className={th}>Content</th>
              <td
                className={td}
                aria-label='renderedContent'
                dangerouslySetInnerHTML={{
                  __html: details.content.rendered,
                }}
              />
            </tr>
            {details.content.fileInfo && (
              <tr>
                <th className={th}>Attached file</th>
                <td className={td}>
                  <FileViewRedirectButton
                    fileInfo={details.content.fileInfo}
                    teamId={actionLog.team.id}
                    exerciseId={exerciseId}
                    inInstructor
                  />
                </td>
              </tr>
            )}
          </tr>
        </>
      )
    default:
      throw new Error(`unknown actionLog type: ${actionLog.type}`)
  }
}

interface ActionDetailProps {
  actionLog: ActionLog
}

const ActionDetail: FC<ActionDetailProps> = ({ actionLog }) => {
  const { exercise } = useContext(ExerciseContext)
  const formatTimestamp = useFormatTimestamp()

  return (
    <div>
      <table>
        <tbody>
          <tr>
            <th className={th}>Action type</th>
            <td className={td}>{actionLog.type}</td>
          </tr>
          <tr>
            <th className={th}>Timestamp</th>
            <td className={td}>{formatTimestamp(actionLog.timestamp)}</td>
          </tr>
          <tr>
            <th className={th}>Team</th>
            <td className={td}>{actionLog.team.name}</td>
          </tr>
          {getContent(actionLog, exercise.id)}
        </tbody>
      </table>
    </div>
  )
}

export default ActionDetail
