import { OPEN_COMPOSE_EVENT_TYPE } from '@/email/EmailFormOverlay'
import type { ActionLog } from '@inject/graphql/fragments/ActionLog.generated'
import type { CustomInjectDetails } from '@inject/graphql/fragments/CustomInjectDetails.generated'
import type { EmailDetails } from '@inject/graphql/fragments/EmailDetails.generated'
import type { InjectDetails } from '@inject/graphql/fragments/InjectDetails.generated'
import type { QuestionnaireDetails } from '@inject/graphql/fragments/QuestionnaireDetails.generated'
import type { ToolDetails } from '@inject/graphql/fragments/ToolDetails.generated'
import { createRef, useEffect, type FC } from 'react'
import CustomInjectContent from './CustomInjectContent'
import EmailContent from './EmailContent'
import InjectContent from './InjectContent'
import InstructorQuestionnaireContent from './InstructorQuestionnaireContent'
import ToolContent from './ToolContent'
import TraineeQuestionnaireContent from './TraineeQuestionnaireContent'

interface ContentProps {
  actionLog: ActionLog
  exerciseId: string
  teamId: string
  inInstructor: boolean
}

const Content: FC<ContentProps> = ({
  actionLog,
  exerciseId,
  teamId,
  inInstructor,
}) => {
  const injectRef = createRef<HTMLDivElement>()
  useEffect(() => {
    if (injectRef.current) {
      injectRef.current.querySelectorAll('a[href^=mailto]').forEach(x => {
        ;(x as HTMLLinkElement).onclick = e => {
          e.preventDefault()
          window.dispatchEvent(
            new CustomEvent(OPEN_COMPOSE_EVENT_TYPE, {
              detail: {
                email: (x as HTMLLinkElement).href.slice(7), // to remove mailto:!
              },
            })
          )
        }
      })
    }
  })
  switch (actionLog.type) {
    case 'INJECT':
      return (
        <InjectContent
          details={actionLog.details as InjectDetails}
          teamId={teamId}
          exerciseId={exerciseId}
          inInstructor={inInstructor}
          ref={injectRef}
        />
      )
    case 'CUSTOM_INJECT':
      return (
        <CustomInjectContent
          details={actionLog.details as CustomInjectDetails}
          teamId={teamId}
          exerciseId={exerciseId}
          inInstructor={inInstructor}
          ref={injectRef}
        />
      )
    case 'TOOL':
      return (
        <ToolContent
          details={actionLog.details as ToolDetails}
          teamId={teamId}
          exerciseId={exerciseId}
          inInstructor={inInstructor}
          ref={injectRef}
        />
      )
    case 'FORM':
      return inInstructor ? (
        <InstructorQuestionnaireContent
          details={actionLog.details as QuestionnaireDetails}
          teamId={teamId}
        />
      ) : (
        <TraineeQuestionnaireContent
          details={actionLog.details as QuestionnaireDetails}
          teamId={teamId}
        />
      )
    case 'EMAIL':
      return (
        <EmailContent
          details={actionLog.details as EmailDetails}
          teamId={teamId}
          exerciseId={exerciseId}
          inInstructor={inInstructor}
          ref={injectRef}
        />
      )
    default:
      throw new Error(`unknown actionLog type: ${actionLog.type}`)
  }
}

export default Content
