import { Divider, NonIdealState } from '@blueprintjs/core'
import { css } from '@emotion/css'
import { useLiveQuery } from 'dexie-react-hooks'
import { memo, type FC } from 'react'
import LearningActivityForm from '../LearningActivityForm'
import { getLearningActivityById } from '../indexeddb/operations'
import type { LearningActivityInfo } from '../indexeddb/types'
import { LearningActivityType } from '../indexeddb/types'
import EmailTemplateForm from './EmailTemplateForm'
import ToolResponseForm from './ToolResponseForm'

const divider = css`
  margin: 1.5rem 0 1rem;
`

interface LearningActivitySpecificationProps {
  learningActivityId: number
}

const LearningActivitySpecification: FC<LearningActivitySpecificationProps> = ({
  learningActivityId,
}) => {
  const activity = useLiveQuery(
    () => getLearningActivityById(learningActivityId),
    [learningActivityId],
    null
  ) as LearningActivityInfo

  if (!activity) {
    return (
      <NonIdealState
        icon='low-voltage-pole'
        title='No activity'
        description='Activity not found'
      />
    )
  }

  return (
    <div>
      <div>
        <p>
          <b>Name:</b> {activity.name}
        </p>
        <p>
          <b>Description:</b> {activity.description}
        </p>
        <p>
          <b>Type:</b> {activity.type}
        </p>
        <LearningActivityForm
          learningActivity={activity}
          learningObjectiveId={activity.learningObjectiveId}
          buttonProps={{
            text: 'Edit activity',
            icon: 'edit',
            style: { marginRight: '1rem' },
          }}
        />
      </div>
      <Divider className={divider} />
      {activity.type === LearningActivityType.TOOL && (
        <ToolResponseForm learningActivityId={learningActivityId} />
      )}
      {activity.type === LearningActivityType.EMAIL && (
        <EmailTemplateForm learningActivityId={learningActivityId} />
      )}
    </div>
  )
}

export default memo(LearningActivitySpecification)
