import {
  Divider,
  NonIdealState,
  Tab,
  Tabs,
  TabsExpander,
} from '@blueprintjs/core'
import { css } from '@emotion/css'
import { useLiveQuery } from 'dexie-react-hooks'
import { memo, useState, type FC } from 'react'
import InjectForm from '../InjectForm'
import { getInjectInfoById } from '../indexeddb/operations'
import type { InjectInfo } from '../indexeddb/types'
import { InjectType } from '../indexeddb/types'
import ConnectionsForm from './ConnectionsForm'
import EmailInjectForm from './EmailInjectForm'
import InformationInjectForm from './InformationInjectForm'
import OverlayForm from './OverlayForm'
import QuestionnaireForm from './QuestionnaireForm'

const divider = css`
  margin: 1.5rem 0 1rem;
`

interface InjectSpecificationProps {
  injectInfoId: number
}

const InjectSpecification: FC<InjectSpecificationProps> = ({
  injectInfoId,
}) => {
  const injectInfo = useLiveQuery(
    () => getInjectInfoById(injectInfoId),
    [injectInfoId],
    null
  ) as InjectInfo
  const [changed, setChanged] = useState(false)

  if (!injectInfo) {
    return (
      <NonIdealState
        icon='low-voltage-pole'
        title='No inject'
        description='Inject not found'
      />
    )
  }

  return (
    <div>
      <div style={{ marginBottom: '1rem' }}>
        <p>
          <b>Name:</b> {injectInfo.name}
        </p>
        <p>
          <b>Description:</b> {injectInfo.description}
        </p>
        <p>
          <b>Type:</b> {injectInfo.type}
        </p>
        <InjectForm
          inject={injectInfo}
          buttonProps={{
            text: 'Edit inject',
            icon: 'edit',
            style: { marginRight: '1rem' },
          }}
        />
      </div>
      <Divider className={divider} />
      <Tabs>
        <Tab
          id='parameters'
          title='Parameters'
          panel={
            <>
              {injectInfo.type === InjectType.EMAIL && (
                <EmailInjectForm
                  injectInfoId={injectInfoId}
                  changed={changed}
                  onChangedChange={value => setChanged(value)}
                />
              )}
              {injectInfo.type === InjectType.INFORMATION && (
                <InformationInjectForm
                  injectInfoId={injectInfoId}
                  changed={changed}
                  onChangedChange={value => setChanged(value)}
                />
              )}
              {injectInfo.type === InjectType.QUESTIONNAIRE && (
                <QuestionnaireForm
                  injectInfoId={injectInfoId}
                  changed={changed}
                  onChangedChange={value => setChanged(value)}
                />
              )}
            </>
          }
        />
        <Tab
          id='connections'
          title='Connections'
          panel={
            <ConnectionsForm
              injectInfoId={injectInfoId}
              injectType={injectInfo.type}
              changed={changed}
              onChangedChange={value => setChanged(value)}
            />
          }
        />
        <Tab
          id='overlay'
          title='Overlay'
          panel={
            <OverlayForm
              injectInfoId={injectInfoId}
              changed={changed}
              onChangedChange={value => setChanged(value)}
            />
          }
        />
        <TabsExpander />
      </Tabs>
    </div>
  )
}

export default memo(InjectSpecification)
