import useEditorStorage from '@/editor/useEditorStorage'
import { Section, SectionCard } from '@blueprintjs/core'
import { useLiveQuery } from 'dexie-react-hooks'
import { memo, useEffect, useState } from 'react'
import { db } from '../indexeddb/db'
import { InjectType } from '../indexeddb/types'
import EmailChannelForm from './EmailChannelForm'
import ExerciseDurationForm from './ExerciseDurationForm'
import FinalMilestoneForm from './FinalMilestoneForm'
import FormChannelForm from './FormChannelForm'
import InfoChannelForm from './InfoChannelForm'
import ToolChannelForm from './ToolChannelForm'

const FinalInformationForm = () => {
  const emailAddressesCount = useLiveQuery(
    () => db.emailAddresses.count(),
    [],
    0
  )
  const toolsCount = useLiveQuery(() => db.tools.count(), [], 0)
  const questionnairesCount = useLiveQuery(
    () => db.injectInfos.where({ type: InjectType.QUESTIONNAIRE }).count(),
    [],
    0
  )
  const [config, setConfig] = useEditorStorage()
  const [exerciseDuration, setExerciseDuration] = useState(
    config?.exerciseDuration || 0
  )
  const [showExerciseTime, setShowExerciseTime] = useState(
    config?.showExerciseTime || false
  )
  const [finalMilestone, setFinalMilestone] = useState(
    config?.finalMilestone || 0
  )
  const [emailChannelName, setEmailChannelName] = useState(
    config?.emailChannelName || ''
  )
  const [emailBetweenTeams, setEmailBetweenTeams] = useState(
    config?.emailBetweenTeams || false
  )
  const [customEmailSuffix, setCustomEmailSuffix] = useState(
    config?.customEmailSuffix || ''
  )
  const [formChannelName, setFormChannelName] = useState(
    config?.formChannelName || ''
  )
  const [infoChannelName, setInfoChannelName] = useState(
    config?.infoChannelName || ''
  )
  const [toolChannelName, setToolChannelName] = useState(
    config?.toolChannelName || ''
  )

  useEffect(() => {
    setConfig(prevConfig => ({
      ...prevConfig,
      exerciseDuration,
      showExerciseTime,
      finalMilestone,
      emailChannelName,
      emailBetweenTeams,
      customEmailSuffix,
      formChannelName,
      infoChannelName,
      toolChannelName,
    }))
  }, [
    exerciseDuration,
    showExerciseTime,
    finalMilestone,
    emailChannelName,
    emailBetweenTeams,
    customEmailSuffix,
    formChannelName,
    infoChannelName,
    toolChannelName,
  ])

  return (
    <div>
      <Section title='Exercise details'>
        <SectionCard>
          <ExerciseDurationForm
            exerciseDuration={exerciseDuration}
            showExerciseTime={showExerciseTime}
            onExerciseDurationChange={(value: number) =>
              setExerciseDuration(value)
            }
            onShowExerciseTimeChange={(value: boolean) =>
              setShowExerciseTime(value)
            }
          />
          <FinalMilestoneForm
            finalMilestone={finalMilestone}
            onFinalMilestoneChange={(value: number) => setFinalMilestone(value)}
          />
        </SectionCard>
      </Section>
      <Section title='Channel details'>
        <SectionCard>
          <InfoChannelForm
            infoChannelName={infoChannelName}
            onInfoChannelNameChange={(value: string) =>
              setInfoChannelName(value)
            }
          />
          {toolsCount > 0 && (
            <ToolChannelForm
              toolChannelName={toolChannelName}
              onToolChannelNameChange={(value: string) =>
                setToolChannelName(value)
              }
            />
          )}
          {emailAddressesCount > 0 && (
            <EmailChannelForm
              emailChannelName={emailChannelName}
              emailBetweenTeams={emailBetweenTeams}
              customEmailSuffix={customEmailSuffix}
              onEmailChannelNameChange={(value: string) =>
                setEmailChannelName(value)
              }
              onEmailBetweenTeamsChange={(value: boolean) =>
                setEmailBetweenTeams(value)
              }
              onCustomEmailSuffixChange={(value: string) =>
                setCustomEmailSuffix(value)
              }
            />
          )}
          {questionnairesCount > 0 && (
            <FormChannelForm
              formChannelName={formChannelName}
              onFormChannelNameChange={(value: string) =>
                setFormChannelName(value)
              }
            />
          )}
        </SectionCard>
      </Section>
    </div>
  )
}

export default memo(FinalInformationForm)
