import { formatTimestamp } from '@/analyst/utilities'
import Label from '@/users/UsersSelection/Label'
import { Button, Colors, Dialog, DialogBody } from '@blueprintjs/core'
import { Cross } from '@blueprintjs/icons'
import { css } from '@emotion/css'
import type { Definition } from '@inject/graphql/fragment-types'
import { dialog, dialogBody } from '@inject/shared/css/dialog'
import { breakWord } from '@inject/shared/css/textOverflow'
import { useState, type FC } from 'react'

const table = css`
  table-layout: fixed;
  width: 100%;
  text-align: left;

  th,
  td {
    padding: 0.5rem;
    vertical-align: top;
  }

  th {
    width: 14ch;
  }
  td {
    ${breakWord}
  }

  ul {
    padding-left: 1rem;
    margin: 0;
  }
`

interface InfoButtonProps {
  definition: Definition
}

const InfoButton: FC<InfoButtonProps> = ({ definition }) => {
  const [dialogOpen, setDialogOpen] = useState(false)

  const { name, version, roles, channels, uploadedBy, timestampCreated } =
    definition

  return (
    <>
      <Button
        icon='info-sign'
        active={dialogOpen}
        onClick={() => setDialogOpen(true)}
      />
      <Dialog
        isOpen={dialogOpen}
        onClose={() => setDialogOpen(false)}
        title={<span title={`Definition ${name}`}>{`Definition ${name}`}</span>}
        className={dialog}
      >
        <DialogBody className={dialogBody}>
          <table className={table}>
            <tbody>
              <tr>
                <th>Name:</th>
                <td>{name}</td>
              </tr>
              {definition.description && (
                <tr>
                  <th>Description:</th>
                  <td>{definition.description}</td>
                </tr>
              )}
              <tr>
                <th>Version:</th>
                <td>{version}</td>
              </tr>
              <tr>
                <th>Roles:</th>
                <td>
                  {/* TODO: add Description */}
                  {roles.length ? (
                    roles.map((role, index) => (
                      <div
                        key={role.id}
                      >{`${role.displayName}${index < roles.length - 1 ? ',' : ''}`}</div>
                    ))
                  ) : (
                    <Cross color={Colors.RED3} />
                  )}
                </td>
              </tr>
              <tr>
                <th>Channels:</th>
                <td>
                  {/* TODO: add Description */}
                  {channels.map((channel, index) => (
                    <div
                      key={channel.id}
                      title={channel.description || channel.displayName}
                    >{`${channel.displayName} (${channel.type})${index < channels.length - 1 ? ',' : ''}`}</div>
                  ))}
                </td>
              </tr>
              {definition.prerequisites && (
                <tr>
                  <th>Prerequisites:</th>
                  <td>
                    <ul>
                      {definition.prerequisites.map(prerequisite => (
                        <li key={prerequisite}>{prerequisite}</li>
                      ))}
                    </ul>
                  </td>
                </tr>
              )}
              {definition.targetAudience && (
                <tr>
                  <th>Target audience:</th>
                  <td>{definition.targetAudience}</td>
                </tr>
              )}
              <tr>
                <th>Uploaded by:</th>
                <td>
                  {uploadedBy ? <Label user={uploadedBy} /> : 'a deleted user'}
                </td>
              </tr>
              <tr>
                <th>Uploaded at:</th>
                <td>{formatTimestamp(timestampCreated)}</td>
              </tr>
            </tbody>
          </table>
        </DialogBody>
      </Dialog>
    </>
  )
}

export default InfoButton
