## Deployment using Docker/Nginx

INJECT Exercise Platform provides a simplified deployment method via Docker Compose. 
This method simplifies the deployment by creating necessary settings and pairings between the frontend and backend and also creating a singular endpoint for accessing INJECT Exercise Platform.

### Docker Setup

This solution consists of 3 containers: frontend, backend, Nginx Proxy (as a front-facing router).

Frontend and Backend are setup to communicate under a single endpoint handled by Nginx Proxy. 
The Proxy forwards appropriate requests either to Frontend or Backend.

Deployment is setup with persistent storage under a Docker Compose volume called `backend-data`. This volume contains any data generated by INJECT and current configuration of the platform. At current state the platform can be effectively backed up assuming you backup `backend-data` volume and `INJECT_SECRET_KEY` environment variable.

For HTTPS deployment it's needed to have HTTPS certificates. At current state, there is a prepared configuration and script for deployment via Let's Encrypt certificates. This script necessitates downtime of IXP to fully refresh or instantiate new SSL certificates.

### Environment Variables

All of the technical configuration is done with environment variables. These are prefixed under `INJECT_`, in addition the deployments contain internal environment variables that are automatically inferred from `INJECT_` prefixed environment variables.

#### General Configuration


```
# sets up an appropriate hostname for the Nginx server
INJECT_DOMAIN=inject.localhost

# Secret key for cryptographic signing
INJECT_SECRET_KEY=your-at-least-fifty-or-more-characters-long-secret-key
```


- INJECT_DOMAIN: Defines the hostname for the Nginx server. This should be set to the **domain name** that points to your server's IP address.
- INJECT_SECRET_KEY: Used to provide cryptographic signing. Must be at least 50 characters long. Can be anything.
- INJECT_HOMEPAGE: The URL (link) of the deployed instance fo the INJECT platform with used protocol (http or https). The following URL is an example: _https://my-inject-domain.com_. This link is used in the welcome email.
- INJECT_MAX_UPLOAD_SIZE: Specifies the maximum body size of requests, including file uploads. The unit of this variable is bytes. The default value is set to 10MB (10485760 bytes).

#### Authentication, Authorization and Identification (AAI)

AAI modules ensured user authentication and sending of user credentials via email invitations. 
The emails are sent out using an external SMTP server.

The settings for sending emails is configured via the following environment variables. 

```
# for email sending of AAI mails
INJECT_EMAIL_HOST=
INJECT_EMAIL_PORT=
INJECT_EMAIL_HOST_USER=
INJECT_EMAIL_HOST_PASSWORD=
INJECT_EMAIL_SENDER_ADDRESS=
INJECT_EMAIL_PROTOCOL=
```


- INJECT_EMAIL_HOST: SMTP server address for sending emails.
- INJECT_EMAIL_PORT: Port for the SMTP server. Port must be always set.
- INJECT_EMAIL_HOST_USER: Username for the SMTP server.
- INJECT_EMAIL_HOST_PASSWORD: Password for the SMTP server.
- INJECT_EMAIL_SENDER_ADDRESS: Email address used as the sender for outgoing emails.
- INJECT_EMAIL_PROTOCOL: Preferred protocol for communication with SMTP server. The choices (values) can be _ssl_ or _tls_ (case insensitive). This variable can also be left undefined - no encryption will be used. Please note, that setting of `INJECT_EMAIL_PROTOCOL` does not automatically set the port and port must be always set.
- INJECT_EMAIL_TIMEOUT: Specifies a timeout in seconds for blocking operations tied to emails like the connection attempt to SMTP. The default value is 10 seconds.

In the current release, AAI can be disabled by setting `INJECT_NOAUTH` to any arbitrary value. 
This feature will be deprecated in upcoming release.

#### Logging and Debugging

If you need to enable logging, configure the following environment variable in your .env file:

```
# enable logging to a specific file
# INJECT_LOGS=data/

```

- INJECT_LOGS: Path to the log file. Uncomment and set this variable to enable logging.

#### Host and Connection Settings

These parameters are setup by default in the `.env` file, it's adviced to not modify them unless you want to specifically modify how IXP interacts with Frontend and Backend. Individual Docker compose setups these settings setup accordingly to the set level of communication. 

```
# host parameters that should be set up for the client and server to interact correctly
INJECT_HOST_ADDRESSES=$INJECT_DOMAIN
VITE_HTTPS_HOST=$INJECT_DOMAIN
CORS_ALLOWED_ORIGINS=https://$INJECT_DOMAIN
```

- INJECT_HOST_ADDRESSES: Comma-separated list of addresses where the backend can be reached. Misconfiguration may lead to an inaccessible Backend instance.
- VITE_HTTPS_HOST: Defines the HTTP host for which the frontend application should connect to, the hostname should point to an instance of Backend application.
- CORS_ALLOWED_ORIGINS: Comma-separated list of origins allowed for Cross-Origin Resource Sharing (CORS). In case of HTTPS it's necessary to change the prefixed protocol from `http` to `https`. Implicitly set-up values by backend include: `http://localhost:5173`, `http://127.0.0.1:5173`, `http://localhost:8000`, `http://127.0.0.1:8000`.


## Good practices

At current state it's adviced:
- if there is any manipulation to be done on said containers (ex. restarting, shutting them down, updating, etc.) to ensure that no exercises are running at given point of time.
- to backup data located in `backend-data` volume
- to backup `INJECT_SECRET_KEY`, these keys are critical to ensure proper functioning of AAI, loss of said keys renders logins to be unusable

## Deprecations

At current state:
- AAI can be optionally disabled in the current version. In the future releases, this won't be possible any longer.
- Database driver is SQLite in current release, in the future it will be migrated to a different solution due to known performance issues related to the chosen database backend.
