package cz.fidentis.analyst.scene;

import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshFacetImpl;
import cz.fidentis.analyst.symmetry.Plane;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

/**
 * Drawable plane with the possibility to shift it along the normal and,
 * moreover, to show a "mirror" plane (a plane shifted in the opposite direction).
 * 
 * @author Radek Oslejsek
 * @author Dominik Racek
 */
public class DrawableCuttingPlane extends DrawablePlane {
    
    private double shift = 0.0;
    
    /**
     * Copy constructor.
     * @param drPlane Original plane
     * @throws NullPointException if the input argument is {@code null}
     */
    public DrawableCuttingPlane(DrawableCuttingPlane drPlane) {
        super(drPlane);
        this.shift = drPlane.shift;
    }
    
    /**
     * Constructor.
     * @param facet Mesh facet of the plane
     * @param plane The plane
     */
    public DrawableCuttingPlane(MeshFacet facet, Plane plane) {
        super(facet, plane);
    }
    
    /**
     * Constructor.
     * 
     * @param plane The plane
     * @param midPoint A 3D point, which is projected to the plane and used as a center of the rectangular facet
     * @param width Width
     * @param height Height
     * @throws NullPointerException if the {@code plane} or {@code midPoint} are {@code null}
     * @throws IllegalArgumentException if {@code width} or {@code height} are &lt;= 0
     */
    public DrawableCuttingPlane(Plane plane, Point3d midPoint, double width, double height) {
        super(plane, midPoint, width, height);
    }

    /**
     * Moves the plane in the plane's normal direction.
     * 
     * @param dist Distance. If positive, then the movements is in the direction of the normal vector.
     * Otherwise, the movement is against the normal direction.
     */
    public void shift(double dist) {
        shift += dist;
        
        Vector3d move = getPlane().getNormal();
        move.scale(dist);
        for (int i = 0; i < getFacets().get(0).getNumberOfVertices(); ++i) {
            getFacets().get(0).getVertex(i).getPosition().sub(move);
        }
        
        if (isMirrorPlaneShown()) {
            move.scale(-1.0);
            for (int i = 0; i < getFacets().get(1).getNumberOfVertices(); ++i) {
                getFacets().get(1).getVertex(i).getPosition().sub(move);
            }
        }
    }
    
    /**
     * Returns shifted cutting plane.
     * @return shifted cutting plane.
     */
    @Override
    public Plane getPlane() {
        return new Plane(super.getPlane().getNormal(), super.getPlane().getDistance() + shift);
    }
    
    /**
     * Returns the cutting plane shifted to opposite direction.
     * @return the cutting plane shifted to opposite direction
     */
    public Plane getMirrorPlane() {
        return new Plane(super.getPlane().getNormal(), super.getPlane().getDistance() - shift);
    }
    
    /**
     * Shows/hides the mirror plane (i.e., the cutting plane shifted in the opposite direction
     * @param show Shows if {@code true}, hides otherwise
     */
    public void showMirrorPlane(boolean show) {
        if (show) {
            if (isMirrorPlaneShown()) { // already shown
                return;
            }
            MeshFacetImpl mirror = new MeshFacetImpl(getModel().getFacets().get(0));
            Vector3d move = new Vector3d(super.getPlane().getNormal());
            move.scale(-2.0 * shift);
            for (int i = 0; i < mirror.getNumberOfVertices(); ++i) {
                mirror.getVertex(i).getPosition().sub(move);
            }
            getModel().addFacet(mirror);
        } else if (isMirrorPlaneShown()) {
            getModel().removeFacet(1);
        }
    }
    
    /**
     * Returns {@code true} if the mirror planes are set to be shown.
     * @return {@code true} if the mirror planes are set to be shown.
     */
    public boolean isMirrorPlaneShown() {
        return getModel().getFacets().size() > 1;
    }
}
