package cz.fidentis.analyst.core;

import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import javax.swing.JFormattedTextField;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

/**
 * ComboSlider implements a combination of horizontal slider and input text field.
 * The slider and input field are synchronized automatically.
 * 
 * @author Radek Oslejsek
 */
public abstract class ComboSlider extends JPanel {
 
    private final JSlider slider = new JSlider();
    private final JFormattedTextField inputField = new JFormattedTextField();
    private boolean continousSync;

    public static final int DEFAULT_TEXT_FIELD_WIDTH = 70;
    public static final int DEFAULT_TEXT_FIELD_HEIGHT = 40;
    
    private ChangeListener changeListener = new ChangeListener() {
        @Override
        public void stateChanged(ChangeEvent e) {
            setValueFromSlider();
        }
    };
    
    private MouseListener mouseListener = new MouseAdapter() {
        @Override
        public void mouseReleased(MouseEvent e) {
            setValueFromSlider();
        }
    };
    
    /**
     * Constructor.
     */
    public ComboSlider() {
        initComponents();
        
        setContinousSync(true); // update input field on slider's change and inform external listeners
        
        inputField.addActionListener((ActionEvent ae) -> { // update slider when the input field changed
            setValueFromInputField();
        });
        
    }
    
    /**
     * Connects the specified listener with the input field.
     * The listener is invoked on the input field change, which is affected
     * by the {@link #setContinousSync(boolean)}.
     * Event's source is set to {@code JFormattedTextField}
     *
     * @param listener the action listener to be added
     */
    public synchronized void addInputFieldListener(ActionListener listener) {
        inputField.addActionListener(listener);
    }
    
    /**
     * Connects the specified listener with the slider.
     * Event's source is set to {@code JSlider}
     *
     * @param listener the action listener to be added
     */
    public synchronized void addSliderListener(ActionListener listener) {
        slider.addChangeListener((ChangeEvent e) -> {
            listener.actionPerformed(new ActionEvent(slider, ActionEvent.ACTION_PERFORMED, null));
        });
    }
    
    /**
     * The slider is on the left, followed by ti input field, by default.
     * This method switches the order.
     */
    public void setSliderEast() {
        remove(slider);
        add(slider);
    }
    
    /**
     * If {@code true}, then the input field is updated during the slider move.
     * Otherwise, the input field is updated only on mouse key release.
     * 
     * @param continousSync Whether to update input field continuously.
     */
    public final void setContinousSync(boolean continousSync) {
        this.continousSync = continousSync;
        if (this.continousSync) {
            slider.removeMouseListener(mouseListener);
            slider.addChangeListener(changeListener);
        } else {
            slider.addMouseListener(mouseListener);
            slider.removeChangeListener(changeListener);
        }
    }

    /**
     * If {@code true}, then the input field is updated during the slider move.
     * Otherwise, the input field is updated only on mouse key release.
     * 
     * @return {@code true} if the slider and input field are synchronized continuously. 
     */
    public boolean isContinousSync() {
        return continousSync;
    }
    
    public JSlider getSlider() {
        return slider;
    }
    
    public JFormattedTextField getInputField() {
        return inputField;
    }
    
    /**
     * Reads the value of the slider, scales it into the range of the input field, 
     * updates the input field and triggers input field change event.
     */
    protected abstract void setValueFromSlider();
    
    /**
     * Reads the value of the input field, scales it into the range of the slider, 
     * and updates the slider. No change event is triggered.
     */
    protected abstract void setValueFromInputField();
    
    protected final void initComponents() {
        inputField.setPreferredSize(new Dimension(DEFAULT_TEXT_FIELD_WIDTH, DEFAULT_TEXT_FIELD_HEIGHT));
        add(slider);
        add(inputField);
    }
    
}
