package cz.fidentis.analyst.batch;

import cz.fidentis.analyst.Logger;
import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.core.ControlPanelAction;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.face.events.HumanFaceEvent;
import cz.fidentis.analyst.face.events.HumanFaceListener;
import cz.fidentis.analyst.core.ProgressDialog;
import java.awt.event.ActionEvent;
import java.beans.PropertyChangeEvent;
import java.util.Arrays;
import javax.swing.JOptionPane;
import javax.swing.JTabbedPane;
import javax.swing.SwingWorker;

/**
 * Action listener for batch registration phase.
 * 
 * @author Radek Oslejsek
 */
public class BatchAction extends ControlPanelAction implements HumanFaceListener {
    
    private final BatchPanel controlPanel;
    private HumanFace avgFace = null;
    private int avgFaceIndex = -1;
    
    /**
     * Constructor.
     *
     * @param canvas OpenGL canvas
     * @param topControlPanel Top component for placing control panels
     */
    public BatchAction(Canvas canvas, JTabbedPane topControlPanel) {
        super(canvas, topControlPanel);
        this.controlPanel = new BatchPanel(this);
        
        // Place control panel to the topControlPanel
        topControlPanel.addTab(controlPanel.getName(), controlPanel.getIcon(), controlPanel);
        topControlPanel.addChangeListener(e -> {
            // If the registration panel is focused...
            if (((JTabbedPane) e.getSource()).getSelectedComponent() instanceof BatchPanel) {
                // TODO
            }
        });
        topControlPanel.setSelectedComponent(controlPanel); // Focus registration panel
        
        // Be informed about changes in faces perfomed by other GUI elements
        //getPrimaryDrawableFace().getHumanFace().registerListener(this);
        //getSecondaryDrawableFace().getHumanFace().registerListener(this);
    }

    @Override
    public void actionPerformed(ActionEvent ae) {
        String action = ae.getActionCommand();
         switch (action) {
            case BatchPanel.ACTION_COMMAND_COMPUTE_ICP:
                computeRegistrationAndTemplate();
                break;
            case BatchPanel.ACTION_COMMAND_COMPUTE_SIMILARITY:
                 computeSimilarity();
                 break;
            default:
         }
        renderScene();
    }

    @Override
    public void acceptEvent(HumanFaceEvent event) {
        // NOTHING TO DO
    }

    private void computeRegistrationAndTemplate() {
        ProgressDialog progressBar = new ProgressDialog(controlPanel, "ICP registration and average face computation");
        
        if (controlPanel.showIcpPreview() && this.avgFaceIndex != -1) {
            getScene().showDrawableFace(avgFaceIndex, false); // hide temporarily
        }
        
        IcpTask task = new IcpTask(
                progressBar, 
                controlPanel, 
                controlPanel.showIcpPreview() ? getCanvas() : null);
        
        task.addPropertyChangeListener((PropertyChangeEvent evt) -> { // when done ...
            if ("state".equals(evt.getPropertyName()) && (SwingWorker.StateValue.DONE.equals(evt.getNewValue()))) {
                avgFace = task.getAverageFace();
                if (avgFaceIndex == -1) { // first successful computation
                    avgFaceIndex = getScene().getFreeIndex();
                }
                getScene().setDrawableFace(avgFaceIndex, avgFace); // either shows or removes the face
                getScene().setFaceAsPrimary(avgFaceIndex);
                getScene().showDrawableFace(avgFaceIndex, true);
                renderScene();
            }
        });
        
        progressBar.runTask(task);
    } 
    
    private void computeSimilarity() {
        
        ProgressDialog progressBar = new ProgressDialog(controlPanel, "Similarity computation");
        final SimilarityTask task;
        
        switch (controlPanel.getSimilarityStrategy()) {
            case BatchPanel.SIMILARITY_COMPLETE_HD:
                task = new CompleteHausdorffDistTask(progressBar, controlPanel);
                break;
            case BatchPanel.SIMILARITY_APPROX_HD:
                if (avgFace == null) {
                    JOptionPane.showMessageDialog(
                            controlPanel, 
                            "Compute average face first",
                            "No average face",
                            JOptionPane.WARNING_MESSAGE);
                    return;
                }
                task = new ApproxHausdorffDistTask(progressBar, controlPanel, avgFace);
                break;
            default:
                return;
        }
        
        task.addPropertyChangeListener((PropertyChangeEvent evt) -> { // when done ...
            if ("state".equals(evt.getPropertyName()) && (SwingWorker.StateValue.DONE.equals(evt.getNewValue()))) {
                double[][] result = task.getSimilarities();
                Logger.print(Arrays.deepToString(result));
                // TO DO ...
            }
        });
        
        progressBar.runTask(task);
    } 
    
}
