package cz.fidentis.analyst;

import cz.fidentis.analyst.face.HumanFace;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;


/**
 * This class encapsulates data for a "comparison project", i.e., a project 
 * enabling analysts to compare and investigate human faces.
 *
 * @author Matej Kovar
 */
public class Project {
    
    private List<HumanFace> faces = new ArrayList<>();
    
    /**
     * Returns list of HumanFace secondary faces
     * 
     * @return list of secondary faces
     */
    public List<HumanFace> getFaces() {
        return Collections.unmodifiableList(faces);
    }

    /**
     * Sets new list of secondary faces
     * 
     * @param secondaryFaces list of HumanFace which will be new list of 
     * secondary faces
     * @throws IllegalArgumentException if one of faces from argument is null
     */
    public void setFaces(List<HumanFace> secondaryFaces) {
        this.faces.clear();
        for (int i = 0; i < secondaryFaces.size(); i++) {
            if (secondaryFaces.get(i) == null) {
                throw new IllegalArgumentException("One of faces is null");
            }
        }
        this.faces.addAll(secondaryFaces);
    }
    
    /**
     * Adds new face to faces
     * 
     * @param face HumanFace which will be added to list of secondary faces
     * @throws IllegalArgumentException when argument face is null
     */
    public void addFace(HumanFace face) {
        if (face == null) {
            throw new IllegalArgumentException("Face is null");
        }
        this.faces.add(face);
    }
    
    /**
     * Removes HumanFace from faces
     * 
     * @param face HumanFace which will be removed from faces
     * @throws IllegalArgumentException when argument face is null
     */
    public void removeFace(HumanFace face) {
        if (face == null) {
            throw new IllegalArgumentException("Face is null");
        }
        
        for (int i = 0; i < faces.size(); i++) {
            if (faces.get(i).equals(face)) {
                faces.remove(i);
            }
        } 
    }
    
    /**
     * Removes faces which are sent to this function by list of HumanFace
     * from faces
     * 
     * @param faces List of HumanFace faces which should be removed from
     * faces
     */
    public void removeSelected(List<HumanFace> faces) {
        for (int i = 0; i < faces.size(); i++) {
            this.removeFace(faces.get(i));
        }
    }
    
    
    /**
     * 
     * @param name String name of the model (face)
     * @return face from this project faces, if not in project than returns 
     * null
     */
    public HumanFace getFaceByName(String name) {
        for (HumanFace face : faces) {
            if (face.getShortName().equals(name)) {
                return face;
            }
        }
        return null;
    }

}
