package cz.fidentis.analyst.distance;

import cz.fidentis.analyst.core.ControlPanel;
import cz.fidentis.analyst.core.ControlPanelBuilder;
import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.feature.FeaturePointType;
import cz.fidentis.analyst.scene.DrawableFeaturePoints;
import cz.fidentis.analyst.symmetry.SymmetryPanel;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.DoubleSummaryStatistics;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

/**
 * Control panel for Hausdorff distance.
 * 
 * @author Radek Oslejsek
 * @author Daniel Schramm
 */
public class DistancePanel extends ControlPanel {
    
    /*
     * Mandatory design elements
     */
    public static final String ICON = "distance28x28.png";
    public static final String NAME = "Hausdorff distance";
    
    /*
     * External actions
     */
    public static final String ACTION_COMMAND_SHOW_HIDE_HEATMAP = "show-hide heatmap";
    public static final String ACTION_COMMAND_SET_DISTANCE_STRATEGY = "set strategy";
    public static final String ACTION_COMMAND_RELATIVE_ABSOLUTE_DIST = "switch abosulte-relative distance";
    public static final String ACTION_COMMAND_WEIGHTED_DISTANCE = "switch weighted distance on/off";
    public static final String ACTION_COMMAND_FEATURE_POINT_HIGHLIGHT = "highlight feature point with color";
    public static final String ACTION_COMMAND_FEATURE_POINT_RESIZE = "set size of feature point";
    public static final String ACTION_COMMAND_DISTANCE_RECOMPUTE = "recompute the Hausdorff distance";
    
    /*
     * Configuration of panel-specific GUI elements
     */
    public static final String STRATEGY_POINT_TO_POINT = "Point to point";
    public static final String STRATEGY_POINT_TO_TRIANGLE = "Point to triangle";
    
    private final Map<FeaturePointType, JLabel> featurePointStats;
    private final JLabel avgHD, avgHDWeight, maxHD, maxHDWeight, minHD, minHDWeight;
    
    /**
     * Constructor.
     * @param action Action listener
     */
    public DistancePanel(ActionListener action, List<FeaturePoint> featurePoints) {
        this.setName(NAME);
        
        final ControlPanelBuilder builder = new ControlPanelBuilder(this);
        
        builder.addCaptionLine("Computation options:");
        builder.addLine();
        
        builder.addCheckBoxOptionLine(
                null, 
                "Relative distance", 
                false,
                createListener(action, ACTION_COMMAND_RELATIVE_ABSOLUTE_DIST)
        );
        builder.addLine();
        
        builder.addOptionText("Distance strategy");
        builder.addComboBox(
                List.of(STRATEGY_POINT_TO_POINT, STRATEGY_POINT_TO_TRIANGLE),
                createListener(action, ACTION_COMMAND_SET_DISTANCE_STRATEGY)
        );
        builder.addGap();
        builder.addLine();
        
        final JPanel featurePointsPanel = new JPanel();
        final ControlPanelBuilder fpBuilder = new ControlPanelBuilder(featurePointsPanel);
        featurePointStats = new HashMap<>(featurePoints.size());
        for (int i = 0; i < featurePoints.size(); i++) {
            final FeaturePoint featurePoint = featurePoints.get(i);
            
            final JCheckBox checkBox = fpBuilder.addCheckBox(
                    false,
                    createListener(action, ACTION_COMMAND_DISTANCE_RECOMPUTE)
            );
            checkBox.setText(featurePoint.getFeaturePointType().getName());
            checkBox.addActionListener(createListener(action, ACTION_COMMAND_FEATURE_POINT_HIGHLIGHT, i));

            final JTextField sliderInput = fpBuilder.addSliderOptionLine(null, null, 100, null);
            sliderInput.setText(ControlPanelBuilder.doubleToStringLocale(DrawableFeaturePoints.DEFAULT_SIZE));
            sliderInput.postActionEvent(); // Set correct position of slider
            sliderInput.addActionListener(new AbstractAction() {
                private final ActionListener listener = createListener(action, ACTION_COMMAND_DISTANCE_RECOMPUTE);
                
                @Override
                public void actionPerformed(ActionEvent ae) {
                    if (!checkBox.isSelected()) {
                        return;
                    }
                    listener.actionPerformed(ae); // Recompute only if the feature point is selected
                }
            });
            sliderInput.addActionListener(createListener(action, ACTION_COMMAND_FEATURE_POINT_RESIZE, i));
            
            fpBuilder.addGap();
            featurePointStats.put(featurePoint.getFeaturePointType(), fpBuilder.addLabelLine(null));
            fpBuilder.addGap();
            
            fpBuilder.addLine();
        }
        builder.addScrollPane(featurePointsPanel)
                .setBorder(BorderFactory.createTitledBorder("Feature points"));
        builder.addLine();
        
        builder.addCaptionLine("Visualization options:");
        builder.addLine();
        builder.addCheckBoxOptionLine(
                null, 
                "Show Hausdorff distance", 
                false,
                createListener(action, ACTION_COMMAND_SHOW_HIDE_HEATMAP)
        ); 
        builder.addLine();
        
        builder.addCheckBoxOptionLine(
                null,
                "Weighted Hausdorff distance",
                false,
                createListener(action, ACTION_COMMAND_WEIGHTED_DISTANCE)
        );
        builder.addLine();
        
        builder.addCaptionLine("Hausdorff distance:");
        builder.addLine();
        
        builder.addOptionText("");
        builder.addOptionText("Hausdorff distance");
        builder.addOptionText("Weighted Hasudorff distance");
        builder.addLine();
        
        builder.addOptionText("Average");
        avgHD = builder.addLabelLine("");
        builder.addGap();
        builder.addGap();
        avgHDWeight = builder.addLabelLine("");
        builder.addLine();
        builder.addOptionText("Maximum");
        maxHD = builder.addLabelLine("");
        builder.addGap();
        builder.addGap();
        maxHDWeight = builder.addLabelLine("");
        builder.addLine();
        builder.addOptionText("Minimum");
        minHD = builder.addLabelLine("");
        builder.addGap();
        builder.addGap();
        minHDWeight = builder.addLabelLine("");
        builder.addLine();
        
        builder.addVerticalStrut();
    }
    
    @Override
    public ImageIcon getIcon() {
        return getStaticIcon();
    }
    
    /**
     * Static implementation of the {@link #getIcon()} method.
     * 
     * @return Control panel icon
     */
    public static ImageIcon getStaticIcon() {
        return new ImageIcon(SymmetryPanel.class.getClassLoader().getResource("/" + ICON));
    }
    
    /**
     * Updates GUI elements that display statistical data about the calculated Hausdorff distance.
     * 
     * @param hausdorffDistance Statistical data of the ordinary Hausdorff distance
     * @param weightedHausdorffDistance Statistical data of the weighted Hausdorff distance
     */
    public void updateHausdorffDistanceStats(DoubleSummaryStatistics hausdorffDistance,
            DoubleSummaryStatistics weightedHausdorffDistance) {
        avgHD.setText(Double.toString(hausdorffDistance.getAverage()));
        avgHDWeight.setText(Double.toString(weightedHausdorffDistance.getAverage()));
        
        maxHD.setText(Double.toString(hausdorffDistance.getMax()));
        maxHDWeight.setText(Double.toString(weightedHausdorffDistance.getMax()));
        
        minHD.setText(Double.toString(hausdorffDistance.getMin()));
        minHDWeight.setText(Double.toString(weightedHausdorffDistance.getMin()));
    }
    
    /**
     * Updates GUI elements that display the weights of feature points
     * used to calculate the weighted Hausdorff distance.
     * 
     * @param featurePointWeights Map of feature point types and their weights
     */
    public void updateFeaturePointWeights(Map<FeaturePointType, Double> featurePointWeights) {
        featurePointStats.forEach((fpType, fpWeightLabel) -> {
            final Double fpWeight = featurePointWeights.get(fpType);
            fpWeightLabel.setText(fpWeight == null ? "" : Double.toString(fpWeight));
        });
    }

}
