package cz.fidentis.analyst.mesh.core;

import javax.vecmath.Vector3d;

/**
 * MeshPoint represents a point with position, normal, and texture coordinates
 *
 * @author Matej Lukes
 */
public class MeshPoint {
    private final Vector3d position, normal, texCoord;

    /**
     * constructor of meshPoint
     *
     * @param position position of MeshPoint
     * @param normal   normal of MeshPoint
     * @param texCoord coordinates in texture
     */
    public MeshPoint(Vector3d position, Vector3d normal, Vector3d texCoord) {
        if (position == null) {
            throw new NullPointerException("position cannot be null");
        }

        this.position = position;
        this.normal = normal;
        this.texCoord = texCoord;
    }

    /**
     * copy constructor of meshPoint
     * @param meshPoint copied meshPoint
     */
    public MeshPoint(MeshPoint meshPoint) {
        this.position = new Vector3d(meshPoint.position);
        this.normal = new Vector3d(meshPoint.normal);
        this.texCoord = new Vector3d(meshPoint.texCoord);

    }

    /**
     * @return normal
     */
    public Vector3d getNormal() {
        return normal;
    }

    /**
     * @return position
     */
    public Vector3d getPosition() {
        return position;
    }

    /**
     * @return texture coordinates
     */
    public Vector3d getTexCoord() {
        return texCoord;
    }

    /**
     * returns new instance of MeshPoint with subtracted position
     *
     * @param subtrahend position to be subtracted
     * @return subtracted MeshPoint
     */
    public MeshPoint subtractPosition(MeshPoint subtrahend) {
        return subtractPosition(subtrahend.position);
    }

    /**
     * returns new instance of MeshPoint with subtracted position
     *
     * @param subtrahend position to be subtracted
     * @return subtracted MeshPoint
     */
    public MeshPoint subtractPosition(Vector3d subtrahend) {
        Vector3d newPosition = new Vector3d(position);
        newPosition.sub(subtrahend);
        return new MeshPoint(new Vector3d(newPosition), normal, new Vector3d(texCoord));
    }

    /**
     * returns new instance of MeshPoint with added position
     *
     * @param addend position to be added
     * @return added MeshPoint
     */
    public MeshPoint addPosition(MeshPoint addend) {
        return addPosition(addend.position);
    }

    /**
     * returns new instance of MeshPoint with added position
     *
     * @param addend position to be added
     * @return added MeshPoint
     */
    public MeshPoint addPosition(Vector3d addend) {
        Vector3d newPosition = new Vector3d(position);
        newPosition.add(addend);
        return new MeshPoint(new Vector3d(newPosition), normal, new Vector3d(texCoord));
    }

    /**
     * returns new instance of MeshPoint with subtracted normal
     *
     * @param subtrahend normal to be subtracted
     * @return subtracted MeshPoint
     */
    public MeshPoint subtractNormal(MeshPoint subtrahend) {
        return subtractNormal(subtrahend.normal);
    }

    /**
     * returns new instance of MeshPoint with subtracted normal
     *
     * @param subtrahend normal to be subtracted
     * @return subtracted MeshPoint
     */
    public MeshPoint subtractNormal(Vector3d subtrahend) {
        Vector3d newNormal = new Vector3d(normal);
        newNormal.sub(subtrahend);
        newNormal.normalize();
        return new MeshPoint(new Vector3d(position), newNormal, new Vector3d(texCoord));
    }

    /**
     * returns new instance of MeshPoint with added normal
     *
     * @param addend normal to be added
     * @return added MeshPoint
     */
    public MeshPoint addNormal(MeshPoint addend) {
        return addNormal(addend.normal);
    }

    /**
     * returns new instance of MeshPoint with added normal
     *
     * @param addend normal to be added
     * @return added MeshPoint
     */
    public MeshPoint addNormal(Vector3d addend) {
        Vector3d newNormal = new Vector3d(normal);
        newNormal.add(addend);
        newNormal.normalize();
        return new MeshPoint(new Vector3d(position), newNormal, new Vector3d(texCoord));
    }

    /**
     * @param obj compared object
     * @return true if positions, normals and texture coordinates are equal, false otherwise
     */
    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof MeshPoint)) {
            return false;
        }

        MeshPoint meshPointObj = (MeshPoint) obj;
        return this.position.equals(meshPointObj.position)
                && this.normal.equals(meshPointObj.normal)
                && this.texCoord.equals(meshPointObj.texCoord);
    }

    /**
     * returns hash of MeshPoint
     *
     * @return hash
     */
    @Override
    public int hashCode() {
        return position.hashCode() + normal.hashCode() + texCoord.hashCode();
    }
}
