package cz.fidentis.analyst.mesh.io;
/**
 * @author Natália Bebjaková
 * <p>
 * Creates new directory and exports model to .obj file in this created directory
 * Exports model's vertices and normals, if there are some
 */

import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshModel;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Locale;

/**
 * @author Natalia Bebjakova
 */
public class MeshModelExporter {
    /**
     * Helping representation of triangle in symmetry estimate
     *
     * @author Natalia Bebjakova
     */
    private class Triangle {
        private int vertex1;
        private int vertex2;
        private int vertex3;

        /**
         * Creates new triangle
         *
         * @param v1 first vertex
         * @param v2 second vertex
         * @param v3 third vertex
         */
        Triangle(int v1, int v2, int v3) {
            this.vertex1 = v1;
            this.vertex2 = v2;
            this.vertex3 = v3;
        }

        /**
         * @return first vertex of triangle
         */
        public int getVertex1() {
            return vertex1;
        }

        /**
         * @param vertex1 new vertex
         */
        public void setVertex1(int vertex1) {
            this.vertex1 = vertex1;
        }

        /**
         * @return second vertex of triangle
         */
        public int getVertex2() {
            return vertex2;
        }

        /**
         * @param vertex2 new vertex
         */
        public void setVertex2(int vertex2) {
            this.vertex2 = vertex2;
        }

        /**
         * @return third vertex of triangle
         */
        public int getVertex3() {
            return vertex3;
        }

        /**
         * @param vertex3 new vertex
         */
        public void setVertex3(int vertex3) {
            this.vertex3 = vertex3;
        }

        /**
         * Triangles are same if they have same vertices
         *
         * @param obj other triangle
         * @return true if two triangles are same
         */
        @Override
        public boolean equals(Object obj) {
            if (!(obj instanceof Triangle)) {
                return false;
            }
            Triangle t = (Triangle) obj;
            return ((this.vertex1 == t.vertex1) || (this.vertex1 == t.vertex2) || (this.vertex1 == t.vertex3)) &&
                    ((this.vertex2 == t.vertex1) || (this.vertex2 == t.vertex2) || (this.vertex2 == t.vertex3)) &&
                    ((this.vertex3 == t.vertex1) || (this.vertex3 == t.vertex2) || (this.vertex3 == t.vertex3));
        }

        /**
         * @return hascode of the triangle
         */
        @Override
        public int hashCode() {
            return (this.vertex1 * 100 ^ this.vertex2 * 100 ^ this.vertex3);
        }
    }

    /**
     * Model to be exported
     */
    private MeshModel model;

    /**
     * @param model model to be exported.
     */
    public MeshModelExporter(MeshModel model) {
        this.model = model;
    }

    /**
     * @param exportFile exported file name
     * @throws IOException Error
     */
    public void exportModelToObj(File exportFile) throws IOException {
        for (MeshFacet facet : model.getFacets()) {
            exportFacetToObj(facet, exportFile);
        }
    }

    /**
     * Export model to OBJ file.
     *
     * @param facet      Facet of the model to be exported, so far every model has one
     * @param exportFile file for exporting.
     * @throws IOException Error
     */
    public void exportFacetToObj(MeshFacet facet, File exportFile) throws IOException {
        int formatIndex = exportFile.getName().lastIndexOf(".");
        String fileName; //name that is writen to file

        if (formatIndex < 0) {
            fileName = exportFile.getName();
        } else {
            fileName = exportFile.getName().substring(0, formatIndex);
        }

        File exportDirectory = new File(exportFile.getParent() + File.separator + fileName);
        exportDirectory.mkdir(); //creates new directory

        exportFile = new File(exportDirectory + File.separator + fileName + ".obj");

        try (BufferedWriter out = new BufferedWriter(new FileWriter(exportFile))) {

            out.write("mtllib " + fileName + ".mtl" + "\n");

            DecimalFormatSymbols otherSymbols = new DecimalFormatSymbols(Locale.getDefault());
            otherSymbols.setDecimalSeparator('.');  //dot as separator for decimal numbers
            DecimalFormat df = new DecimalFormat("#.####", otherSymbols);

            //writes vertices of the facet to file
            for (int j = 0; j < facet.getNumberOfVertices(); j++) {
                out.write("v " + df.format(facet.getVertices().get(j).getPosition().x) + " "
                        + df.format(facet.getVertices().get(j).getPosition().y) + " "
                        + df.format(facet.getVertices().get(j).getPosition().z) + "\n");
            }
            out.write("#" + facet.getNumberOfVertices() + " vertices" + "\n");

            //detects if first vertex has normal
            boolean hasAllNormals = facet.getVertices().get(0).getNormal() != null;

            //writes normals if there are any 
            for (int i = 0; i < facet.getNumberOfVertices(); i++) {
                if (facet.getVertex(i).getNormal() != null) {
                    out.write("vn " + df.format(facet.getVertices().get(i).getNormal().x) + " "
                            + df.format(facet.getVertices().get(i).getNormal().y) + " "
                            + df.format(facet.getVertices().get(i).getNormal().z) + "\n");
                }
            }

            //computes triangles of facet
            Triangle[] triangles = new Triangle[facet.getCornerTable().getSize() / 3];
            for (int i = 0; i < facet.getCornerTable().getSize(); i += 3) {
                Triangle t = new Triangle(facet.getCornerTable().getRow(i).getVertexIndex(),
                        facet.getCornerTable().getRow(i + 1).getVertexIndex(),
                        facet.getCornerTable().getRow(i + 2).getVertexIndex());
                triangles[(i / 3)] = t;
            }

            //writes triangles of facet
            writeTrianglesOfFacet(triangles, out, facet, hasAllNormals);
            out.write("\n");
        }
    }

    private void writeTrianglesOfFacet(Triangle[] triangles,
                                       BufferedWriter out,
                                       MeshFacet facet,
                                       boolean hasAllNormals) throws IOException {
        for (Triangle triangle : triangles) {
            out.write("f ");
            if (facet.getVertex(triangle.getVertex1()).getNormal() != null && hasAllNormals) {
                out.write((triangle.getVertex1() + 1) + "//" + (triangle.getVertex1() + 1) + " "
                        + (triangle.getVertex2() + 1) + "//" + (triangle.getVertex2() + 1) + " "
                        + (triangle.getVertex3() + 1) + "//" + (triangle.getVertex3() + 1) + "\n");
            } else {
                out.write((triangle.getVertex1() + 1) + " " + (triangle.getVertex2() + 1) +
                        " " + (triangle.getVertex3() + 1) + "\n");
            }
        }
    }
}

