package cz.fidentis.analyst.procrustes;

import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.procrustes.exceptions.ProcrustesAnalysisException;
import cz.fidentis.analyst.procrustes.utils.ProcrustesAnalysisUtils;
import org.ejml.simple.SimpleMatrix;
import org.ejml.simple.SimpleSVD;

import java.util.List;
import javax.vecmath.Vector3f;


/**
 * @author Jakub Kolman
 */
public class ProcrustesAnalysis {

    private List<FeaturePoint> orderedFeaturePointList1;
    private List<FeaturePoint> orderedFeaturePointList2;

    private ProcrustesAnalysisFaceModel featurePointModel1;
    private ProcrustesAnalysisFaceModel featurePointModel2;

    private Vector3f centroid;

    /**
     * @param fpList1
     * @param vertices1
     * @param fpList2
     * @param vertices2
     * @throws ProcrustesAnalysisException
     */
    public ProcrustesAnalysis(
            List<FeaturePoint> fpList1, List<Vector3f> vertices1,
            List<FeaturePoint> fpList2, List<Vector3f> vertices2) throws ProcrustesAnalysisException {
        if (fpList1.size() != fpList2.size()) {
            throw new ProcrustesAnalysisException("Lists of feature points do not have the same size");
        }

        orderedFeaturePointList1 = ProcrustesAnalysisUtils.sortListByFeaturePointType(fpList1);
        orderedFeaturePointList2 = ProcrustesAnalysisUtils.sortListByFeaturePointType(fpList2);

        if (!featurePointTypesEquivalence(fpList1, fpList2)) {
            throw new ProcrustesAnalysisException("Lists of feature points do not have the same feature point types");
        }

        // Might be sufficient to send as parameter orderedFpList instead of fpList1 and continue to work
        // only with faceModels in rest of the class
        featurePointModel1 = new ProcrustesAnalysisFaceModel(fpList1, vertices1);
        featurePointModel2 = new ProcrustesAnalysisFaceModel(fpList2, vertices2);

        centroid = ProcrustesAnalysisUtils.findCenteroidOfFeaturePoints(orderedFeaturePointList1);

        superImpose(featurePointModel1, featurePointModel2);


    }

    /**
     * Imposes two face models (lists of feature points and vertices) over each other
     *
     * @param featurePointModel1
     * @param featurePointModel2
     */
    private void superImpose(ProcrustesAnalysisFaceModel featurePointModel1,
                             ProcrustesAnalysisFaceModel featurePointModel2) {

        // moves vertices of face models
        featurePointModel1.setVertices(moveVertices(centroid, featurePointModel1.getVertices()));
        featurePointModel1.setVertices(moveVertices(centroid, featurePointModel2.getVertices()));

        rotate(featurePointModel1, featurePointModel2);
    }

    /**
     * By rotation of matrices solves orthogonal procrustes problem
     *
     * @param featurePointModel1
     * @param featurePointModel2
     */
    private void rotate(ProcrustesAnalysisFaceModel featurePointModel1,
                        ProcrustesAnalysisFaceModel featurePointModel2) {
        // There is no reason trying to rotate less than 3 elements
        if (this.orderedFeaturePointList1.size() < 3) {
            return;
        }
        SimpleMatrix primaryMatrix = ProcrustesAnalysisUtils.createFeaturePointMatrix(orderedFeaturePointList2);
        SimpleMatrix transposedMatrix = ProcrustesAnalysisUtils.createFeaturePointMatrix(
                orderedFeaturePointList1).transpose();

        SimpleMatrix multipliedMatrix = primaryMatrix.mult(transposedMatrix);
        SimpleSVD<SimpleMatrix> singularValueDecomposition = multipliedMatrix.svd();
        SimpleMatrix transposedU = singularValueDecomposition.getU().transpose();
        SimpleMatrix r = singularValueDecomposition.getV().mult(transposedU);
        primaryMatrix = primaryMatrix.mult(r);

        featurePointModel2.setFeaturePointsMap(
                ProcrustesAnalysisUtils.createFeaturePointMapFromMatrix(primaryMatrix, featurePointModel2));

        rotateVertices(featurePointModel2, primaryMatrix);
    }

    /**
     * Rotates every vertex of model by given matrix
     *
     * @param model
     * @param matrix
     */
    private void rotateVertices(ProcrustesAnalysisFaceModel model, SimpleMatrix matrix) {
        if (model.getVertices() != null) {
            int i = 0;
            for (Vector3f v : model.getVertices()) {
                v.set(ProcrustesAnalysisUtils.rotateVertex(v, matrix));
            }
        }
    }

    /**
     * Calculates new vertices adjusted to the centroid by method {@see findCenteroidOfFeaturePoints},
     * so moved featurepoints would correspond with the same place on face as they did before.
     *
     * @param centroid
     * @param vertices
     */
    private List<Vector3f> moveVertices(Vector3f centroid, List<Vector3f> vertices) {
        if (vertices != null && centroid != null) {
            List<Vector3f> movedVertices = null;
            for (Vector3f v : vertices) {
                float x = v.x + centroid.x;
                float y = v.y + centroid.y;
                float z = v.z + centroid.z;
                movedVertices.add(new Vector3f(x, y, z));
            }
            return movedVertices;
        }
        throw new ProcrustesAnalysisException("Could not compute vertices locations after moving the centroid from model to feature points");

    }

    /**
     * Creates sorted feature point lists and compares them whether they have
     * the same feature point types.
     *
     * @param fpList1
     * @param fpList2
     * @return
     */
    private boolean featurePointTypesEquivalence(List<FeaturePoint> fpList1, List<FeaturePoint> fpList2) {
        orderedFeaturePointList1 = ProcrustesAnalysisUtils.sortListByFeaturePointType(fpList1);
        orderedFeaturePointList2 = ProcrustesAnalysisUtils.sortListByFeaturePointType(fpList2);

        return ProcrustesAnalysisUtils.checkfeaturePointsType(orderedFeaturePointList1, orderedFeaturePointList2);
    }

    /**
     * Calculate scaling ratio of how much the appropriate object corresponding
     * to the second feature point list has to be scale up or shrunk.
     * <p>
     * If returned ratioValue is greater 1 then it means that the second object
     * should be scaled up ratioValue times. If returned ratioValue is smaller 1
     * than the second object should be shrunk.
     *
     * @return ratioValue
     */
    private double calculateScalingValue() {
        double[] distancesOfList1 = ProcrustesAnalysisUtils.calculateMeanDistancesFromOrigin(orderedFeaturePointList1);
        double[] distancesOfList2 = ProcrustesAnalysisUtils.calculateMeanDistancesFromOrigin(orderedFeaturePointList2);

        double[] ratioArray = new double[distancesOfList1.length];
        double ratioValue = 0;

        for (int i = 0; i < distancesOfList1.length; i++) {
            ratioArray[i] += distancesOfList1[i] / distancesOfList2[i];
        }

        for (double ratio : ratioArray) {
            ratioValue += ratio;
        }

        return ratioValue / distancesOfList1.length;
    }

}
