package cz.fidentis.analyst.canvas.toolbar;

import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.face.events.HumanFaceEvent;
import cz.fidentis.analyst.face.events.HumanFaceListener;
import cz.fidentis.analyst.face.events.SymmetryPlaneChangedEvent;
import cz.fidentis.analyst.scene.DrawableFace;
import cz.fidentis.analyst.scene.DrawablePlane;
import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import javax.swing.AbstractAction;
import javax.swing.ImageIcon;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.JToggleButton;
import javax.swing.event.ChangeEvent;
import javax.swing.plaf.metal.MetalToggleButtonUI;
import org.openide.util.NbBundle;

/**
 * Canvas toolbox for scenes with two faces. 
 * It enables to show-hide faces in the scene, show-hide feature points,
 * and to change the transparency of face couple.
 * 
 * @author Richard Pajersky
 * @author Radek Oslejsek
 */
public class SceneToolboxFaceToFace extends JPanel implements HumanFaceListener {
    
    private static final String SYMMETRY_BUTTON_ICON = "symmetry28x28.png";
    private static final String PRIMARY_FACE_ICON = "primaryFace28x28.png";
    private static final String SECONDARY_FACE_ICON = "secondaryFace28x28.png";
    private static final String FEATURE_POINTS_ICON = "fps28x28.png";
    private static final String DISTANCE_BUTTON_ICON = "distance28x28.png";
    
    private static final int TRANSPARENCY_RANGE = 50;
    
    private JToggleButton priLandButton;
    private JToggleButton secLandButton;
    private JToggleButton priFaceButton;
    private JToggleButton secFaceButton;
    private JToggleButton priSymmetryButton;
    private JToggleButton secSymmetryButton;
    private JSlider       slider;
    private JToggleButton secDistButton;
    
    private final Canvas canvas;
    
    /**
     * Constructor.
     * @param canvas Rendering canvas
     */
    public SceneToolboxFaceToFace(Canvas canvas) {
        this.canvas = canvas;
        initComponents();  // buttons are turned on by default
        
        // be informed if something changes in the faces -- see acceptEvent()
        canvas.getPrimaryFace().registerListener(this);
        canvas.getSecondaryFace().registerListener(this);
        
        // Change inital state:

        // show the symmetry planes automatically, but only after they are computed
        priSymmetryButton.setSelected(true); 
        priSymmetryButton.setEnabled(false); 
        secSymmetryButton.setSelected(true); 
        secSymmetryButton.setEnabled(false); 
        
        // currently, FPs (landmarks) are either presented or not
        priLandButton.setSelected(false);
        if (canvas.getPrimaryFace().hasFeaturePoints()) {
            canvas.getScene().getDrawableFeaturePoints(canvas.getScene().getPrimaryFaceSlot()).show(false);
        } else {
            priLandButton.setEnabled(false);
        }
        secLandButton.setSelected(false);
        if (canvas.getSecondaryFace().hasFeaturePoints()) {
            canvas.getScene().getDrawableFeaturePoints(canvas.getScene().getSecondaryFaceSlot()).show(false);
        } else {
            secLandButton.setEnabled(false);
        }
        
        secDistButton.setSelected(false);
        canvas.getScene().getDrawableFace(canvas.getScene().getSecondaryFaceSlot()).setRenderHeatmap(false);
        
        slider.setValue(30);
        canvas.getScene().getDrawableFace(canvas.getScene().getPrimaryFaceSlot()).setTransparency(30/(float)TRANSPARENCY_RANGE);
        canvas.getScene().getDrawableFace(canvas.getScene().getSecondaryFaceSlot()).setTransparency(1);
    }
    
    
    @Override
    public void acceptEvent(HumanFaceEvent event) {
        if (event instanceof SymmetryPlaneChangedEvent) { 
            DrawablePlane dp = null;
            JToggleButton button = null;
            HumanFace face = event.getFace();

            if (face.equals(canvas.getPrimaryFace())) {
                dp = canvas.getScene().getDrawableSymmetryPlane(canvas.getScene().getPrimaryFaceSlot());
                button = priSymmetryButton;
            } else if (face.equals(canvas.getSecondaryFace())) {
                dp = canvas.getScene().getDrawableSymmetryPlane(canvas.getScene().getSecondaryFaceSlot());
                button = secSymmetryButton;
            }

            if (dp != null) { // the symmetry plane is included in the scene
                button.setEnabled(true);
                dp.show(button.isSelected());
            } else {
                button.setEnabled(false);
            }
        }
    }
    
    private JToggleButton initButton(Color color, String icon, String tooltip) {
        JToggleButton button = new JToggleButton();
        button.setBorder(Canvas.BUTTONS_BORDER);
        if (color != null) {
            button.setBackground(null); // default color
            button.setUI(new MetalToggleButtonUI() {
                @Override
                protected Color getSelectColor() {
                    return color;
                }
            });
        }
        button.setIcon(new ImageIcon(getClass().getResource("/" + icon)));
        button.setFocusable(false);
        button.setSelected(true);
        button.setToolTipText(NbBundle.getMessage(SceneToolboxFaceToFace.class, tooltip));
        
        return button;
    }

    private void initComponents() {
        JPanel panel = new JPanel();
        panel.setLayout(new FlowLayout());
        add(panel);
        
        priSymmetryButton = initButton(null, SYMMETRY_BUTTON_ICON, "SceneToolboxFaceToFace.symmetryButton.tooltip");
        panel.add(priSymmetryButton);
        priSymmetryButton.addActionListener(new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                boolean onOff = ((JToggleButton) e.getSource()).isSelected();
                canvas.getScene().getDrawableSymmetryPlane(canvas.getScene().getPrimaryFaceSlot()).show(onOff);
                canvas.renderScene();
            }
        });

        priLandButton = initButton(null, FEATURE_POINTS_ICON, "SceneToolboxFaceToFace.landButton.tooltip");
        panel.add(priLandButton);
        priLandButton.addActionListener(new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                boolean onOff = ((JToggleButton) e.getSource()).isSelected();
                canvas.getScene().getDrawableFeaturePoints(canvas.getScene().getPrimaryFaceSlot()).show(onOff);
                canvas.renderScene();
            }
        });

        priFaceButton = initButton(DrawableFace.SKIN_COLOR_PRIMARY, PRIMARY_FACE_ICON, "SceneToolboxFaceToFace.faceButton.tooltip");
        panel.add(priFaceButton);
        priFaceButton.addActionListener(new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                boolean onOff = ((JToggleButton) e.getSource()).isSelected();
                canvas.getScene().showDrawableFace(canvas.getScene().getPrimaryFaceSlot(), onOff);
                canvas.renderScene();
            }
        });
        
        /////////////////////////
        slider = new JSlider();
        slider.setMajorTickSpacing(TRANSPARENCY_RANGE/2);
        slider.setMaximum(TRANSPARENCY_RANGE*2);
        slider.setMinorTickSpacing(TRANSPARENCY_RANGE/2);
        slider.setPaintTicks(false);
        slider.setToolTipText(org.openide.util.NbBundle.getMessage(SceneToolboxFaceToFace.class, "SceneToolboxFaceToFace.slider.tooltip"));
        slider.setValue(TRANSPARENCY_RANGE);
        slider.setCursor(new java.awt.Cursor(java.awt.Cursor.HAND_CURSOR));
        panel.add(slider);
        
        slider.addChangeListener((ChangeEvent e) -> {
            int val = ((JSlider) e.getSource()).getValue();
            if (val <= TRANSPARENCY_RANGE) {
                canvas.getScene().getDrawableFace(canvas.getScene().getPrimaryFaceSlot())
                        .setTransparency(val/(float)TRANSPARENCY_RANGE);
                canvas.getScene().getDrawableFace(canvas.getScene().getSecondaryFaceSlot()).setTransparency(1);
            } else if (val > TRANSPARENCY_RANGE) {
                canvas.getScene().getDrawableFace(canvas.getScene().getPrimaryFaceSlot()).setTransparency(1);
                canvas.getScene().getDrawableFace(canvas.getScene().getSecondaryFaceSlot())
                        .setTransparency((2 * TRANSPARENCY_RANGE - val) / (float)TRANSPARENCY_RANGE);
            }
            canvas.renderScene();
        });
        
        /////////////////////////
        secFaceButton = initButton(DrawableFace.SKIN_COLOR_SECONDARY, SECONDARY_FACE_ICON, "SceneToolboxFaceToFace.faceButton.tooltip");
        panel.add(secFaceButton);
        secFaceButton.addActionListener(new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                boolean onOff = ((JToggleButton) e.getSource()).isSelected();
                canvas.getScene().showDrawableFace(canvas.getScene().getSecondaryFaceSlot(), onOff);
                canvas.renderScene();
            }
        });
        
        secLandButton = initButton(null, FEATURE_POINTS_ICON, "SceneToolboxFaceToFace.landButton.tooltip");
        panel.add(secLandButton);
        secLandButton.addActionListener(new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                boolean onOff = ((JToggleButton) e.getSource()).isSelected();
                canvas.getScene().getDrawableFeaturePoints(canvas.getScene().getSecondaryFaceSlot()).show(onOff);
                canvas.renderScene();
            }
        });
        
        secSymmetryButton = initButton(null, SYMMETRY_BUTTON_ICON, "SceneToolboxFaceToFace.symmetryButton.tooltip");
        panel.add(secSymmetryButton);
        secSymmetryButton.addActionListener(new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                boolean onOff = ((JToggleButton) e.getSource()).isSelected();
                canvas.getScene().getDrawableSymmetryPlane(canvas.getScene().getSecondaryFaceSlot()).show(onOff);
                canvas.renderScene();
            }
        });

        secDistButton = initButton(null, DISTANCE_BUTTON_ICON, "SceneToolboxFaceToFace.distButton.tooltip");
        panel.add(secDistButton);
        secDistButton.addActionListener(new AbstractAction() { 
            @Override
            public void actionPerformed(ActionEvent e) {
                boolean onOff = ((JToggleButton) e.getSource()).isSelected();
                canvas.getScene().getDrawableFace(canvas.getScene().getSecondaryFaceSlot()).setRenderHeatmap(onOff);
                canvas.renderScene();
            }
        });
    }
}
