package cz.fidentis.analyst.scene;

import com.jogamp.opengl.GL2;
import cz.fidentis.analyst.mesh.core.MeshRectangleFacet;
import cz.fidentis.analyst.symmetry.Plane;
import cz.fidentis.analyst.visitors.mesh.BoundingBox.BBox;
import javax.vecmath.Point3d;

/**
 * A plane to be shown as a rectangular mesh facet.
 * The rectangular shape is generated at run-time from the associated plane.
 * 
 * @author Radek Oslejsek
 * @author Dominik Racek
 */
public class DrawablePlane extends Drawable {
    
    private Plane plane;
    private final BBox bbox;
    private boolean showBoundingBox;
    
    
    /**
     * Constructor.
     * 
     * @param plane plane
     * @param bbox bounding box used to estimate cutting pale shape (rectangle)
     * @param showBBox whether to render bounding box
     */
    public DrawablePlane(Plane plane, BBox bbox, boolean showBBox) {
        this.plane = plane;
        this.bbox = bbox;
        this.showBoundingBox = showBBox;
    }

    /**
     * Copy constructor.
     * @param drPlane Original plane
     * @throws NullPointException if the input argument is {@code null}
     */
    public DrawablePlane(DrawablePlane drPlane) {
        super(drPlane);
        this.plane = new Plane(drPlane.getPlane());
        this.bbox = drPlane.bbox;
        this.showBoundingBox = drPlane.showBoundingBox;
    }
    
    public Plane getPlane() {
        return plane;
    }
    
    /**
     * Returns {@code true} if the bounding box is to be rendered.
     * @return {@code true} if the bounding box is to be rendered.
     */
    public boolean isShownBBobx() {
        return this.showBoundingBox;
    }
    
    /**
     * Switches on/off bounding box rendering.
     * 
     * @param show If {@code true], then the bounding box is rendered as well
     */
    public void setShowBBox(boolean show) {
        this.showBoundingBox = show;
    }
    
    protected void setPlane(Plane plane) {
        this.plane = plane;
    }

    @Override
    protected void renderObject(GL2 gl) {
        renderObject(gl, getPlane().getMesh(bbox));
    }
    
    protected void renderObject(GL2 gl, MeshRectangleFacet facet) {
        gl.glBegin(GL2.GL_TRIANGLES); //vertices are rendered as triangles
        // get the normal and tex coords indicies for face i  
        for (int v = 0; v < facet.getCornerTable().getSize(); v++) {
            Point3d vert = facet.getVertices().get(facet.getCornerTable().getRow(v).getVertexIndex()).getPosition();
            gl.glVertex3d(vert.x, vert.y, vert.z);
        }
        gl.glEnd();
        
        if (showBoundingBox) {
            Point3d min = bbox.getMinPoint();
            Point3d max = bbox.getMaxPoint();
            gl.glBegin(GL2.GL_LINES); 

            // bottom rectangle:
            gl.glVertex3d(min.x, min.y, min.z);
            gl.glVertex3d(max.x, min.y, min.z);
            
            gl.glVertex3d(max.x, min.y, min.z);
            gl.glVertex3d(max.x, min.y, max.z);
            
            gl.glVertex3d(max.x, min.y, max.z);
            gl.glVertex3d(min.x, min.y, max.z);
            
            gl.glVertex3d(min.x, min.y, max.z);
            gl.glVertex3d(min.x, min.y, min.z);

            // top rectangle:
            gl.glVertex3d(min.x, max.y, min.z);
            gl.glVertex3d(max.x, max.y, min.z);
            
            gl.glVertex3d(max.x, max.y, min.z);
            gl.glVertex3d(max.x, max.y, max.z);
            
            gl.glVertex3d(max.x, max.y, max.z);
            gl.glVertex3d(min.x, max.y, max.z);
            
            gl.glVertex3d(min.x, max.y, max.z);
            gl.glVertex3d(min.x, max.y, min.z);
            
            // vertical edges:
            gl.glVertex3d(min.x, min.y, min.z);
            gl.glVertex3d(min.x, max.y, min.z);
            
            gl.glVertex3d(max.x, min.y, min.z);
            gl.glVertex3d(max.x, max.y, min.z);
            
            gl.glVertex3d(max.x, min.y, max.z);
            gl.glVertex3d(max.x, max.y, max.z);
            
            gl.glVertex3d(min.x, min.y, max.z);
            gl.glVertex3d(min.x, max.y, max.z);
            
            gl.glEnd();
        }
    }
    
    protected BBox getBBox() {
        return this.bbox;
    }
}
