package cz.fidentis.analyst.registration;

import cz.fidentis.analyst.Logger;
import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.core.ControlPanelAction;
import cz.fidentis.analyst.core.SpinSlider;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.face.HumanFaceUtils;
import cz.fidentis.analyst.face.events.HausdorffDistanceComputed;
import cz.fidentis.analyst.face.events.HumanFaceEvent;
import cz.fidentis.analyst.face.events.HumanFaceListener;
import cz.fidentis.analyst.face.events.HumanFaceTransformedEvent;
import cz.fidentis.analyst.face.events.SymmetryPlaneChangedEvent;
import cz.fidentis.analyst.scene.DrawablePointCloud;
import cz.fidentis.analyst.visitors.mesh.sampling.CurvatureSampling;
import cz.fidentis.analyst.visitors.mesh.sampling.NoSampling;
import cz.fidentis.analyst.visitors.mesh.sampling.PointSampling;
import cz.fidentis.analyst.visitors.mesh.sampling.RandomSampling;
import cz.fidentis.analyst.visitors.mesh.sampling.UniformSpaceSampling;
import java.awt.Color;
import java.awt.event.ActionEvent;
import javax.swing.JFormattedTextField;
import javax.swing.JOptionPane;
import javax.swing.JTabbedPane;
import javax.vecmath.Vector3d;

/**
 * Action listener for the ICP and Procrustes registration.
 * <p>
 * Besides the UX logic, this object stores the parameters and results of 
 * manual, ICP, or Procrustes registration (alignment of human faces) 
 * <p/>
 * <p>
 * This object also serves as {@code HumanFaceListener}.
 * It means that it is invoked whenever one of the faces are changed and then can 
 * react to these changes.
 * <p/>
 * <p>
 * Changes made by this objects are announced to other listeners. 
 * Following events are triggered:
 * <ul>
 * <li>{@code MeshChangedEvent}</li>
 * <li></li>
 * </ul>
 * </p>
 *
 * @author Richard Pajersky
 * @author Radek Oslejsek
 * @author Daniel Schramm
 */
public class RegistrationAction extends ControlPanelAction implements HumanFaceListener  {

    /*
     * Attributes handling the state
     */
    private String strategy = RegistrationPanel.STRATEGY_POINT_TO_POINT;
    private boolean relativeDist = false;
    private boolean heatmapRender = false;
    
    /**
     * Point cloud of undersampled secondary face
     */
    private int pointCloudSlot = -1;
    
    /*
     * Coloring threshold and statistical values of feature point distances:
     */
    private double fpThreshold = 5.0;

    private final RegistrationPanel controlPanel;

    /**
     * Constructor.
     *
     * @param canvas OpenGL canvas
     * @param topControlPanel Top component for placing control panels
     */
    public RegistrationAction(Canvas canvas, JTabbedPane topControlPanel) {
        super(canvas, topControlPanel);
        this.controlPanel = new RegistrationPanel(this);

        // Place control panel to the topControlPanel
        topControlPanel.addTab(controlPanel.getName(), controlPanel.getIcon(), controlPanel);
        topControlPanel.addChangeListener(e -> {
            // If the registration panel is focused...
            if (((JTabbedPane) e.getSource()).getSelectedComponent() instanceof RegistrationPanel) {
                // ... display heatmap and feature points relevant to the registration
                //getCanvas().getScene().setDefaultColors();
                highlightCloseFeaturePoints();
                //getSecondaryDrawableFace().setRenderHeatmap(heatmapRender);
                
                pointCloudSlot = drawPointSamples(
                        getCanvas().getScene().getSecondaryFaceSlot(), 
                        pointCloudSlot, 
                        controlPanel.getIcpUndersamplingStrength());
            } else {
                getScene().setOtherDrawable(pointCloudSlot, null);
                pointCloudSlot = -1;
            }
        });
        topControlPanel.setSelectedComponent(controlPanel); // Focus registration panel
        //calculateHausdorffDistance();
        highlightCloseFeaturePoints();
        
        // Be informed about changes in faces perfomed by other GUI elements
        getPrimaryDrawableFace().getHumanFace().registerListener(this);
        getSecondaryDrawableFace().getHumanFace().registerListener(this);
        
        controlPanel.setEnabledPlanesButton(
                getCanvas().getPrimaryFace().hasSymmetryPlane() &&
                        getCanvas().getSecondaryFace().hasSymmetryPlane()
        );
        
        controlPanel.setEnabledProcrustesButton(
                getCanvas().getPrimaryFace().hasFeaturePoints() &&
                        getCanvas().getSecondaryFace().hasFeaturePoints()
        );
    }

    @Override
    public void actionPerformed(ActionEvent ae) {
        String action = ae.getActionCommand();

        switch (action) {
            case RegistrationPanel.ACTION_COMMAND_APPLY_ICP:
                applyICP();
                highlightCloseFeaturePoints();
                getCanvas().getSecondaryFace().announceEvent(new HumanFaceTransformedEvent(
                        getCanvas().getSecondaryFace(), "", this)
                );
                pointCloudSlot = drawPointSamples(getCanvas().getScene().getSecondaryFaceSlot(), pointCloudSlot, controlPanel.getIcpUndersamplingStrength());
                break;
            case RegistrationPanel.ACTION_COMMAND_MANUAL_TRANSFORMATION_IN_PROGRESS:
                HumanFaceUtils.transformFace(
                        getSecondaryDrawableFace().getHumanFace(),
                        controlPanel.getAndClearManualRotation(),
                        controlPanel.getAndClearManualTranslation(),
                        controlPanel.getAndClearManualScale(),
                        false
                );
                highlightCloseFeaturePoints();
                getCanvas().getSecondaryFace().announceEvent(new HumanFaceTransformedEvent(
                        getCanvas().getSecondaryFace(), "", this, false) // transformation under progress
                );
                break;
            case RegistrationPanel.ACTION_COMMAND_MANUAL_TRANSFORMATION_FINISHED:
                getCanvas().getSecondaryFace().announceEvent(new HumanFaceTransformedEvent(
                        getCanvas().getSecondaryFace(), "", this, true) // finished transformation
                );
                pointCloudSlot = drawPointSamples(getCanvas().getScene().getSecondaryFaceSlot(), pointCloudSlot, controlPanel.getIcpUndersamplingStrength());
                break;
            case RegistrationPanel.ACTION_COMMAND_FP_CLOSENESS_THRESHOLD:
                fpThreshold = ((Number) (((JFormattedTextField) ae.getSource()).getValue())).doubleValue();
                highlightCloseFeaturePoints();
                break;
            case RegistrationPanel.ACTION_COMMAND_PROCRUSTES_APPLY:
                applyProcrustes();
                highlightCloseFeaturePoints();
                getCanvas().getPrimaryFace().announceEvent(new HumanFaceTransformedEvent(
                        getCanvas().getPrimaryFace(), "", this)
                );
                getCanvas().getSecondaryFace().announceEvent(new HumanFaceTransformedEvent(
                        getCanvas().getSecondaryFace(), "", this)
                );
                pointCloudSlot = drawPointSamples(getCanvas().getScene().getSecondaryFaceSlot(), pointCloudSlot, controlPanel.getIcpUndersamplingStrength());
                break;
            case RegistrationPanel.ACTION_COMMAND_ALIGN_SYMMETRY_PLANES:
                alignSymmetryPlanes();
                highlightCloseFeaturePoints();
                getCanvas().getSecondaryFace().announceEvent(new HumanFaceTransformedEvent(
                        getCanvas().getSecondaryFace(), "", this)
                );
                pointCloudSlot = drawPointSamples(getCanvas().getScene().getSecondaryFaceSlot(), pointCloudSlot, controlPanel.getIcpUndersamplingStrength());
                break;
            case RegistrationPanel.ACTION_COMMAND_POINT_SAMPLING_STRENGTH:
                int numSamples = (Integer) ((SpinSlider) ae.getSource()).getValue();
                pointCloudSlot = drawPointSamples(getCanvas().getScene().getSecondaryFaceSlot(), pointCloudSlot, numSamples);
                break;
            case RegistrationPanel.ACTION_COMMAND_POINT_SAMPLING_STRATEGY:
                pointCloudSlot = drawPointSamples(getCanvas().getScene().getSecondaryFaceSlot(), pointCloudSlot, controlPanel.getIcpUndersamplingStrength());
                break;
            default:
                // do nothing
        }

        renderScene();
    }
    
    @Override
    public void acceptEvent(HumanFaceEvent event) {
        if (event.getIssuer() == this) {
            return;
        }
        
        if (event instanceof HausdorffDistanceComputed) { // update stats
            HausdorffDistanceComputed hdEvent = (HausdorffDistanceComputed) event;
            controlPanel.updateHausdorffDistanceStats(
                    hdEvent.getHusdorffDistStats(),
                    hdEvent.getWeightedHusdorffDistStats()
            );
        }
        
        if (event instanceof SymmetryPlaneChangedEvent) {
            controlPanel.setEnabledPlanesButton(
                    getCanvas().getPrimaryFace().hasSymmetryPlane() &&
                            getCanvas().getSecondaryFace().hasSymmetryPlane()
            );
        }
    }
    
    /**
     * Calculates Procrustes analysis.
     * 
     * First it creates object of type ProcrusteAnalasys containing two face models converted
     * to ProcrustesAnalysisFaceModel objects which are required for next analysis step.
     * 
     * In the analysis step, faces are superimposed and rotated.
     * 
     * If the {@code procrustesScalingEnabled} attribute is set to true, then one of the faces is scaled as well.
     */
    protected void applyProcrustes() {
        JOptionPane.showMessageDialog(
                controlPanel,
                "This method is not finished yet (neither feature points nor symmetry planes are transformed properly)",
                "Warning",
                JOptionPane.INFORMATION_MESSAGE);
        
        Logger out = Logger.measureTime();
        
        HumanFaceUtils.alignFeaturePoints(
                getCanvas().getPrimaryFace(), 
                getCanvas().getSecondaryFace(), 
                controlPanel.getScaleParam(), 
                false // drop k-d tree, if exists
        );

        out.printDuration("Procrustes for models with "
                + getCanvas().getPrimaryFace().getMeshModel().getNumVertices()
                + "/"
                + getCanvas().getSecondaryFace().getMeshModel().getNumVertices()
                + " vertices and "
                + getCanvas().getPrimaryFace().getFeaturePoints().size()
                + " feature points"
        );
    }
    
    protected void applyICP() {
        Logger out = Logger.measureTime();
        PointSampling samplingStrategy = getSamplingStrategy();
        
        HumanFaceUtils.alignMeshes(
                getCanvas().getPrimaryFace(),
                getCanvas().getSecondaryFace(), // is transformed
                controlPanel.getMaxIcpIterParam(),
                controlPanel.getScaleParam(),
                controlPanel.getMinIcpErrorParam(),
                samplingStrategy,
                false // drop k-d tree, if exists
        );
        
        out.printDuration("ICP for models with "
                + getCanvas().getPrimaryFace().getMeshModel().getNumVertices()
                + "/"
                + getCanvas().getSecondaryFace().getMeshModel().getNumVertices()
                + " vertices. Downsampling of the secondary face: "
                + samplingStrategy
        );
    }
    
    /**
     * Calculates feature points which are too far away and changes their color
     * to green otherwise set color to default
     */
    private void highlightCloseFeaturePoints() {
        if (getPrimaryDrawableFace() == null) { // scene not yet initiated
            return;
        }

        if (getPrimaryFeaturePoints() == null
                || getSecondaryFeaturePoints() == null
                || getPrimaryFeaturePoints().getFeaturePoints().size() != getSecondaryFeaturePoints().getFeaturePoints().size()) {
            return;
        }

        double fpMaxDist = Double.NEGATIVE_INFINITY;
        double fpMinDist = Double.POSITIVE_INFINITY;
        double distSum = 0.0;
        for (int i = 0; i < getPrimaryFeaturePoints().getFeaturePoints().size(); i++) {
            Vector3d v = new Vector3d(getPrimaryFeaturePoints().getFeaturePoints().get(i).getPosition());
            v.sub(getSecondaryFeaturePoints().getFeaturePoints().get(i).getPosition());
            double distance = v.length();
            if (distance > fpThreshold) {
                getPrimaryFeaturePoints().resetColorToDefault(i);
                getSecondaryFeaturePoints().resetColorToDefault(i);
            } else {
                getPrimaryFeaturePoints().setColor(i, Color.GREEN);
                getSecondaryFeaturePoints().setColor(i, Color.GREEN);
            }
            fpMaxDist = Math.max(fpMaxDist, distance);
            fpMinDist = Math.min(fpMinDist, distance);
            distSum += distance;
        }
        double fpAvgDist = distSum / getPrimaryFeaturePoints().getFeaturePoints().size();
        // to do: show ACF dist
    }

    private void alignSymmetryPlanes() {
        HumanFaceUtils.alignSymmetryPlanes(
                getPrimaryDrawableFace().getHumanFace(),
                getSecondaryDrawableFace().getHumanFace(), // is transformed
                false, // drops k-d tree, if exists
                true   // forbid rotation around normal
        );
    }
    
    private PointSampling getSamplingStrategy() {
        String st = controlPanel.getIcpUdersamplingStrategy();
        double strength = controlPanel.getIcpUndersamplingStrength() / 100.0;
        
        if (st.equals(RegistrationPanel.POINT_SAMPLING_STRATEGIES[0])) {
            return new NoSampling();
        } else if (st.equals(RegistrationPanel.POINT_SAMPLING_STRATEGIES[1])) {
            return new RandomSampling(strength);
        } else if (st.equals(RegistrationPanel.POINT_SAMPLING_STRATEGIES[2])) {
            return new CurvatureSampling(CurvatureSampling.CurvatureAlg.MEAN, strength);
        } else if (st.equals(RegistrationPanel.POINT_SAMPLING_STRATEGIES[3])) {
            return new CurvatureSampling(CurvatureSampling.CurvatureAlg.GAUSSIAN, strength);
        } else if (st.equals(RegistrationPanel.POINT_SAMPLING_STRATEGIES[4])) {
            return new CurvatureSampling(CurvatureSampling.CurvatureAlg.MAX, strength);
        } else if (st.equals(RegistrationPanel.POINT_SAMPLING_STRATEGIES[5])) {
            return new CurvatureSampling(CurvatureSampling.CurvatureAlg.MIN, strength);
        } else if (st.equals(RegistrationPanel.POINT_SAMPLING_STRATEGIES[6])) {
            return new UniformSpaceSampling(strength);
        } else {
            return null;
        }
    }
    
    private int drawPointSamples(int faceSlot, int cloudSlot, int numSamples) {
        HumanFace face = getCanvas().getHumanFace(faceSlot);
        if (face == null) {
            return -1;
        }
        
        PointSampling sampling = getSamplingStrategy();
        sampling.setRequiredSamples(numSamples/100.0);
        face.getMeshModel().compute(sampling);
        
        if (sampling.getClass() == NoSampling.class) { // don't show
            if (cloudSlot != -1) {
                getScene().setOtherDrawable(cloudSlot, null);
            }
            return -1;
        } else {
            if (cloudSlot == -1) {
                cloudSlot = getCanvas().getScene().getFreeSlot();
            }
            getScene().setOtherDrawable(cloudSlot, new DrawablePointCloud(sampling.getSamples()));
            return cloudSlot;
        }
    }
    
    
}
