package cz.fidentis.analyst.tests;

import cz.fidentis.analyst.batch.Stopwatch;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.face.HumanFaceUtils;
import cz.fidentis.analyst.visitors.mesh.HausdorffDistance;
import cz.fidentis.analyst.visitors.mesh.sampling.CurvatureSampling;
import cz.fidentis.analyst.visitors.mesh.sampling.NoSampling;
import cz.fidentis.analyst.visitors.mesh.sampling.PointSampling;
import cz.fidentis.analyst.visitors.mesh.sampling.RandomSampling;
import cz.fidentis.analyst.visitors.mesh.sampling.UniformSpaceSampling;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.stream.Collectors;

/**
 * This class evaluates efficiency (acceleration) and precision of ICP with different 
 * point sampling algorithms and different downsampling strength.
 * 
 * @author Radek Oslejsek
 */
public class IcpDownsampling {
    
    private static final String DATA_DIR = "../../analyst-data-antropologie/_ECA";
    private static final int MAX_SAMPLES = 100;
    
    /**
     * Main method 
     * @param args Input arguments 
     * @throws IOException on IO error
     */
    public static void main(String[] args) throws IOException, ClassNotFoundException, Exception {
        List<Path> faces = Files.list(new File(DATA_DIR).toPath())
                .filter(f -> f.toString().endsWith(".obj"))
                .sorted()
                .limit(MAX_SAMPLES)
                .collect(Collectors.toList());
        
        SortedMap<Double, Stopwatch> efficiency = new TreeMap<>();
        SortedMap<Double, List<Double>> precision = new TreeMap<>();
        
        String alg = "random";
        //String alg = "gaussian";
        //String alg = "uniform space";
        
        int counter = 1;
        for (int i = 0; i < faces.size(); i++) {
            for (int j = i; j < faces.size(); j++) { // starts with "i"!
                if (i != j) { // register only different faces
                    System.out.println(counter + " / " + (faces.size()*faces.size()/2));
                    compareFaces(faces.get(i), faces.get(j), efficiency, precision, alg);
                    printResults(efficiency, precision, counter, alg);
                    counter++;
                }
            }
        }
    }
    
    protected static void printResults(SortedMap<Double, Stopwatch> efficiency, SortedMap<Double, List<Double>> precision, double counter, String sampling) {
        System.out.println();
        System.out.println("Avg. Time (ms) - " + sampling + " sampling:");
        efficiency.entrySet().forEach(e -> {
            System.out.println(e.getKey() + ";" + (e.getValue().getTotalTime() / counter));
        });
        System.out.println();
        System.out.println("Avg. Precision - " + sampling + "  sampling:");
        precision.entrySet().forEach(e -> {
            System.out.println(e.getKey() + ";" 
                    + e.getValue().get(0) + ";" // min
                    + e.getValue().get(1) + ";" // max
                    + (e.getValue().get(2) / counter)); // avg
        });
    }

    protected static void compareFaces(
            Path priFacePath, Path secFacePath, 
            Map<Double, Stopwatch> efficiency, Map<Double, List<Double>> precision,
            String samp) throws IOException {
        
        //double[] percs = new double[]{0.5,1,2,4,6,8,10,15,20,30,40,50,60,70,80,85,90,92,94,96,98,100};
        double[] percs = new double[]{100,90,80,70,60,50,40,30,20,15,10,8,6,4,2,1,0.5};
        
        HumanFace priFace = new HumanFace(priFacePath.toFile());
        priFace.computeKdTree(false);
        
        for (double i: percs) {
            System.out.println("" + i);
            HumanFace secFace = new HumanFace(secFacePath.toFile());
            
            PointSampling sampling;
            switch (samp) {
                case "random":
                    sampling = new RandomSampling();
                    break;
                case "gaussian":
                    sampling = new CurvatureSampling(CurvatureSampling.CurvatureAlg.GAUSSIAN);
                    break;
                case "uniform space":
                    sampling = new UniformSpaceSampling();
                    break;
                default:
                    return;
            }
            
            sampling.setRequiredSamples(i/100.0);
            
            efficiency.computeIfAbsent(i, k-> new Stopwatch("")).start();
            HumanFaceUtils.alignMeshes(
                            priFace,
                            secFace, // is transformed
                            100,  // max iterations
                            false,// scale
                            0.3,  // error
                            (i == 100) ? new NoSampling() : sampling,  // no undersampling
                            false // drop k-d tree, if exists
            );
            efficiency.get(i).stop();
            
            HausdorffDistance hd = new HausdorffDistance(
                    priFace.getKdTree(),
                    HausdorffDistance.Strategy.POINT_TO_POINT,
                    false,  // relative distance
                    true,   // parallel
                    true    // crop
            );
            secFace.getMeshModel().compute(hd);
            
            List<Double> val = precision.computeIfAbsent(i, k -> Arrays.asList(Double.POSITIVE_INFINITY, Double.NEGATIVE_INFINITY, 0.0));
            val.set(0, Math.min(val.get(0), hd.getStats().getAverage()));
            val.set(1, Math.max(val.get(1), hd.getStats().getAverage()));
            val.set(2, val.get(2) + hd.getStats().getAverage());
        }
        System.out.println();
    }
}
