package cz.fidentis.analyst.core;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

/**
 * SpinSlider implements a combination of horizontal slider and input text field (Spinner).
 * The slider and the spinner are synchronized automatically.
 * 
 * @author Radek Oslejsek
 */
public class SpinSlider extends JPanel {
    
    /**
     * SpinSlider type
     * 
     * @author Radek Oslejsek
     */
    public enum ValueType {
        INTEGER,
        DOUBLE,
        PERCENTAGE
    };
    
    private final JSlider slider = new JSlider();
    private JSpinner spinner;
    private ValueType type;
    
    private boolean continousSync;
    
    /**
     * Listener for continuous synchronization of the slider and the spinner.
     * The spinner value is updates whenever the slider is moved.
     */
    private ChangeListener changeListener = new ChangeListener() {
        @Override
        public void stateChanged(ChangeEvent e) {
            JSlider s = (JSlider) e.getSource();
            if (spinner.getClass() == DoubleSpinner.class) {
                Double val = Double.valueOf((Integer) s.getValue());
                val /= ((DoubleSpinner) spinner).getRecomputationFactor();
                spinner.setValue(val);
            } else {
                spinner.setValue(s.getValue());
            }
        }
    };
    
    /**
     * Listener for postponed synchronization of the slider and the spinner.
     * The spinner value remains unchanged until the mouse button is released.
     */
    private MouseListener mouseListener = new MouseAdapter() {
        @Override
        public void mouseReleased(MouseEvent e) {
            JSlider s = (JSlider) e.getSource();
            if (spinner.getClass() == DoubleSpinner.class) {
                Double val = Double.valueOf((Integer) s.getValue());
                val /= ((DoubleSpinner) spinner).getRecomputationFactor();
                spinner.setValue(val);
            } else {
                spinner.setValue(s.getValue());
            }
        }
    };
    
    
    /**
     * Constructor that creates percentage spin-slider.
     * Call {@link #initDouble(double, double, double, int)} 
     * or {@link #initInteger(int, int, int, int)} to change it.
     */
    public SpinSlider() {
        initPercentage(100);
        initComponents();
        setContinousSync(true); // update input field on slider's change and inform external listeners
        slider.setFont(new java.awt.Font("Dialog", 1, 0)); // hide numbers
        
        setBackground(Color.LIGHT_GRAY);
    }
    
    /**
     * Returns current value reflecting the slider's position.
     * Even if the postponed synchronization is turned on (see {@link #setContinousSync(boolean)}),
     * the correct value is computed and returned anyway.
     * <p>
     * Based on the type, the return value is
     * <ul>
     * <li>{@code INTEGER}: An {@code Integer} from set range.</li>
     * <li>{@code DOUBLE}: An {@code Double} from set range.</li>
     * <li>{@code PERCENTAGE}: An {@code Integer} between 0 and 100.</li>
     * </ul>
     * </p>
     * <p>
     * Usage:
     * <ul>
     * <li>After {@code initDouble()}: {@code double var = (Double) spinSlider.getValue()}.</li>
     * <li>After {@code initInteger()}: {@code double var = (Integer) spinSlider.getValue()}.</li>
     * <li>After {@code initPercentage()}: {@code double var = (Integer) spinSlider.getValue()}.</li>
     * </ul>
     * </p>
     * 
     * @return Current value
     */
    public Number getValue() {
        if (this.type == ValueType.DOUBLE) {
            DoubleSpinner ds = (DoubleSpinner) spinner;
            return slider.getValue() / ds.getRecomputationFactor();
        } else {
            return (Number) spinner.getValue();
        }
    }
    
    /**
     * Sets the value. 
     * 
     * @param value A new value
     */
    public void setValue(Number value) {
        spinner.setValue(value);
    }
    
    /**
     * Initializes this spin-slider to integers.
     * <b>Removed all listeners!</b>
     * 
     * @param value Initial value
     * @param minimum Minimum value
     * @param maximum Maximum value
     * @param stepSize Spin step size
     */
    public void initInteger(int value, int min, int max, int stepSize) {
        remove(slider);
        if (spinner != null) {
            remove(spinner);
        }
        
        this.type = ValueType.INTEGER;
        spinner = new JSpinner();
        spinner.setModel(new SpinnerNumberModel(value, min, max, stepSize));
        spinner.setEditor(new JSpinner.NumberEditor(spinner));
        slider.setMinimum(min);
        slider.setMaximum(max);
        slider.setValue(value);
        
        initComponents();
        setValue(value);
    }

    /**
     * Initializes this spin-slider to doubles.
     * <b>Removed all listeners!</b>
     * 
     * @param value Initial value
     * @param minimum Minimum value
     * @param maximum Maximum value
     * @param fractionDigits precision of flouting numbers, i.e., 
     *          the number of digits allowed after the floating dot
     */
    public void initDouble(double value, double min, double max, int fractionDigits) {
        remove(slider);
        if (spinner != null) {
            remove(spinner);
        }
        
        this.type = ValueType.DOUBLE;
        DoubleSpinner ds = new DoubleSpinner(fractionDigits);
        spinner = ds;
        slider.setMinimum((int) (min * ds.getRecomputationFactor()));
        slider.setMaximum((int) (max * ds.getRecomputationFactor()));
        slider.setValue((int) (value * ds.getRecomputationFactor()));
        
        initComponents();
        setValue(value);
    }
    
    /**
     * Initializes this spin-slider to percents.
     * 
     * @param value Initial value between 0 and 100.
     */
    public void initPercentage(int value) {
        initPercentage(value, 0, 100);
    }
    
    /**
     * Initializes this spin-slider to percents in given range.
     * 
     * @param value Initial value between 0 and 100.
     * @param min min value in the range 0..100
     * @param max max value in the range 0..100
     */
    public void initPercentage(int value, int min, int max) {
        if (min > max || min < 0 || max > 100) {
            throw new IllegalArgumentException("min/max: " + min + "/" + max);
        }
        
        remove(slider);
        if (spinner != null) {
            remove(spinner);
        }
        
        this.type = ValueType.PERCENTAGE;
        spinner = new JSpinner();
        spinner.setModel(new SpinnerNumberModel(value, 0, 100, 1));
        spinner.setEditor(new JSpinner.NumberEditor(spinner, "0'%'"));
        slider.setMinimum(min);
        slider.setMaximum(max);
        slider.setValue(value);
        
        initComponents();
        setValue(value);
    }
    
    /**
     * The slider is on the left, followed by ti input field, by default.
     * This method switches the order.
     */
    //public void setSliderEast() {
    //    remove(slider);
    //    add(slider);
    //}
    
    /**
     * If {@code true}, then the spinner is updated continuously during the slider move.
     * Otherwise, the spinner remains unchanged until the mouse key releases.
     * 
     * @param continousSync Whether to update the spinner continuously.
     */
    public final void setContinousSync(boolean continousSync) {
        this.continousSync = continousSync;
        if (this.continousSync) {
            slider.removeMouseListener(mouseListener);
            slider.addChangeListener(changeListener);
        } else {
            slider.addMouseListener(mouseListener);
            slider.removeChangeListener(changeListener);
        }
    }
    
    /**
     * If {@code true}, then the spinner is updated continuously during the slider move.
     * Otherwise, the spinner remains unchanged until the mouse key releases.
     * 
     * @return {@code true} if the slider and the spinner are synchronized continuously. 
     */
    public boolean isContinousSync() {
        return continousSync;
    }
    
    /**
     * Be informed when the spinner's value changes.
     * See also {@link #setContinousSync(boolean)}.
     * Event's source is set to {@code JSpinner}. 
     * For the double type of the spin-slider, it is {@code DoubleSpinner}
     *
     * @param listener the action listener to be added
     */
    public synchronized void addSpinnerListener(ActionListener listener) {
        spinner.addChangeListener((ChangeEvent e) -> {
            listener.actionPerformed(new ActionEvent(spinner, ActionEvent.ACTION_PERFORMED, null));
        });
    }
    
    /**
     * Be informed when the slider changes.
     * See also {@link #setContinousSync(boolean)}.
     * Event's source is set to {@code JSlider}
     *
     * @param listener the action listener to be added
     */
    public synchronized void addSliderListener(ActionListener listener) {
        slider.addChangeListener((ChangeEvent e) -> {
            listener.actionPerformed(new ActionEvent(slider, ActionEvent.ACTION_PERFORMED, null));
        });
    }
    
    @Override
    public void setEnabled(boolean enabled) {
        slider.setEnabled(enabled);
        spinner.setEnabled(enabled);
    }
    
    protected final void initComponents() {
        //this.setLayout(new FlowLayout());
        this.setLayout(new BorderLayout());
        
        add(slider, BorderLayout.LINE_START);
        add(spinner, BorderLayout.LINE_END);
        
        spinner.addChangeListener(new ChangeListener() {
            @Override
            public void stateChanged(ChangeEvent e) {
                JSpinner s = (JSpinner) e.getSource();
                if (spinner.getClass() == DoubleSpinner.class) {
                    Double val = (Double) s.getValue();
                    val *= ((DoubleSpinner) spinner).getRecomputationFactor();
                    slider.setValue(val.intValue());
                    //slider.setValue(((Double) s.getValue()).intValue());
                } else {
                    slider.setValue((Integer) s.getValue());
                }
            }
        });
    }    
}
