package cz.fidentis.analyst.canvas.toolbar;

import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.face.events.HumanFaceEvent;
import cz.fidentis.analyst.face.events.HumanFaceListener;
import cz.fidentis.analyst.face.events.SymmetryPlaneChangedEvent;
import cz.fidentis.analyst.scene.DrawableFace;
import cz.fidentis.analyst.scene.DrawablePlane;
import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import javax.swing.AbstractAction;
import javax.swing.ImageIcon;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.JToggleButton;
import javax.swing.event.ChangeEvent;
import javax.swing.plaf.metal.MetalToggleButtonUI;
import org.openide.util.NbBundle;

/**
 * Canvas toolbox for scenes with a single faces. 
 * It enables to show-hide the face and its feature points,
 * and to change the transparency of the face.
 * 
 * @author Richard Pajersky
 * @author Radek Oslejsek
 */
public class SceneToolboxSingleFace extends JPanel implements HumanFaceListener {
    
    private static final String SYMMETRY_BUTTON_ICON = "symmetry28x28.png";
    private static final String FACE_ICON = "head28x28.png";
    private static final String FEATURE_POINTS_ICON = "fps28x28.png";
    
    private static final int TRANSPARENCY_RANGE = 100;
    
    private JToggleButton priLandButton;
    private JToggleButton priFaceButton;
    private JToggleButton priSymmetryButton;
    private JSlider       slider;
    
    private final Canvas canvas;
    
    /**
     * Constructor.
     * @param canvas Rendering canvas
     */
    public SceneToolboxSingleFace(Canvas canvas) {
        this.canvas = canvas;
        initComponents();
        
        // show the symmetry planes automatically, but only after they are computed
        priSymmetryButton.setSelected(true); 
        priSymmetryButton.setEnabled(false); 
        
        // currently, FPs (landmarks) are either presented or not
        priLandButton.setSelected(false);
        if (canvas.getPrimaryFace() != null) {
            if (canvas.getPrimaryFace().hasFeaturePoints()) {
                canvas.getScene().getDrawableFeaturePoints(canvas.getScene().getPrimaryFaceSlot()).show(false);
            } else {
                priLandButton.setEnabled(false);
            }
        
            // be informed if something changes in the faces -- see acceptEvent()
            canvas.getPrimaryFace().registerListener(this);
        }
    }
    
    @Override
    public void acceptEvent(HumanFaceEvent event) {
        if (event instanceof SymmetryPlaneChangedEvent) { 
            DrawablePlane dp = canvas.getScene().getDrawableSymmetryPlane(canvas.getScene().getPrimaryFaceSlot());
            if (dp != null) { // the symmetry plane is included in the scene
                priSymmetryButton.setEnabled(true);
                dp.show(priSymmetryButton.isSelected());
            } else {
                priSymmetryButton.setEnabled(false);
            }
        }
    }
    
    private JToggleButton initButton(Color color, String icon, String tooltip) {
        JToggleButton button = new JToggleButton();
        button.setBorder(Canvas.BUTTONS_BORDER);
        if (color != null) {
            button.setBackground(null); // default color
            button.setUI(new MetalToggleButtonUI() {
                @Override
                protected Color getSelectColor() {
                    return color;
                }
            });
        }
        button.setIcon(new ImageIcon(getClass().getResource("/" + icon)));
        button.setFocusable(false);
        button.setSelected(true);
        button.setToolTipText(NbBundle.getMessage(SceneToolboxFaceToFace.class, tooltip));
        
        return button;
    }

    private void initComponents() {
        JPanel panel = new JPanel();
        panel.setLayout(new FlowLayout());
        add(panel);
        
        priSymmetryButton = initButton(null, SYMMETRY_BUTTON_ICON, "SceneToolboxFaceToFace.symmetryButton.tooltip");
        panel.add(priSymmetryButton);
        priSymmetryButton.addActionListener(new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                boolean onOff = ((JToggleButton) e.getSource()).isSelected();
                canvas.getScene().getDrawableSymmetryPlane(canvas.getScene().getPrimaryFaceSlot()).show(onOff);
                canvas.renderScene();
            }
        });
        
        priLandButton = initButton(null, FEATURE_POINTS_ICON, "SceneToolboxFaceToFace.landButton.tooltip");
        panel.add(priLandButton);
        priLandButton.addActionListener(new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                boolean onOff = ((JToggleButton) e.getSource()).isSelected();
                canvas.getScene().getDrawableFeaturePoints(canvas.getScene().getPrimaryFaceSlot()).show(onOff);
                canvas.renderScene();
            }
        });
        
        priFaceButton = initButton(DrawableFace.SKIN_COLOR_PRIMARY, FACE_ICON, "SceneToolboxFaceToFace.faceButton.tooltip");
        panel.add(priFaceButton);
        priFaceButton.addActionListener(new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                boolean onOff = ((JToggleButton) e.getSource()).isSelected();
                canvas.getScene().showDrawableFace(canvas.getScene().getPrimaryFaceSlot(), onOff);
                canvas.renderScene();
            }
        });
        

        /////////////////////////
        slider = new JSlider();
        //slider.setMajorTickSpacing(TRANSPARENCY_RANGE/2);
        slider.setMaximum(TRANSPARENCY_RANGE);
        //slider.setMinorTickSpacing(TRANSPARENCY_RANGE/2);
        slider.setPaintTicks(false);
        slider.setToolTipText(org.openide.util.NbBundle.getMessage(SceneToolboxSingleFace.class, "SceneToolboxSingleFace.slider.tooltip"));
        slider.setValue(TRANSPARENCY_RANGE);
        slider.setCursor(new java.awt.Cursor(java.awt.Cursor.HAND_CURSOR));
        panel.add(slider);
        
        slider.addChangeListener((ChangeEvent e) -> {
            int val = ((JSlider) e.getSource()).getValue();
            canvas.getScene().getDrawableFace(canvas.getScene().getPrimaryFaceSlot()).setTransparency(val/100f);
            canvas.renderScene();
        });
        
    }
}
