package cz.fidentis.analyst.symmetry;

import cz.fidentis.analyst.Logger;
import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.core.ControlPanelAction;
import cz.fidentis.analyst.face.events.HumanFaceEvent;
import cz.fidentis.analyst.face.events.HumanFaceListener;
import cz.fidentis.analyst.face.events.MeshChangedEvent;
import cz.fidentis.analyst.face.events.SymmetryPlaneComputed;
import cz.fidentis.analyst.scene.DrawableFace;
import cz.fidentis.analyst.scene.DrawablePlane;

import java.awt.event.ActionEvent;
import javax.swing.JTabbedPane;

/**
 * Action listener for the manipulation with the symmetry plane.
 * 
 * @author Radek Oslejsek
 */
public class SymmetryAction extends ControlPanelAction implements HumanFaceListener  {

    private final SymmetryPanel controlPanel;
    private final JTabbedPane topControlPanel;

    /**
     * Constructor.
     * 
     * @param canvas OpenGL canvas
     * @param topControlPanel Top component for placing control panels
     */
    public SymmetryAction(Canvas canvas, JTabbedPane topControlPanel) {
        super(canvas, topControlPanel);
        this.topControlPanel = topControlPanel;
        //this.controlPanel = new SymmetryPanel(this);
        this.controlPanel = new SymmetryPanel(this);

        // Place control panel to the topControlPanel
        this.topControlPanel.addTab(controlPanel.getName(), controlPanel.getIcon(), controlPanel);
        this.topControlPanel.addChangeListener(e -> {
            // If the symmetry panel is focused...
            if (((JTabbedPane) e.getSource()).getSelectedComponent() instanceof SymmetryPanel) {
                getCanvas().getScene().setDefaultColors();
                getCanvas().getScene().showSymmetryPlanes();
            } else {
                getCanvas().getScene().hideSymmetryPlanes();
            }
        });

        recomputeSymmetryPlane(0);
        recomputeSymmetryPlane(1);
        
        // Be informed about changes in faces perfomed by other GUI elements
        getPrimaryDrawableFace().getHumanFace().registerListener(this);
        if (getSecondaryDrawableFace() != null) {
            getSecondaryDrawableFace().getHumanFace().registerListener(this);
        }
    }
    
    @Override
    public void actionPerformed(ActionEvent ae) {
        String action = ae.getActionCommand();
        
        switch (action) {
            case SymmetryPanel.ACTION_COMMAND_RECOMPUTE: 
                recomputeSymmetryPlane(0);
                recomputeSymmetryPlane(1);
                break;  
            default:
                // do nothing
        }
        renderScene();
    }
    
    @Override
    public void acceptEvent(HumanFaceEvent event) {
        if (event instanceof MeshChangedEvent) { // recompte symmetry plane
            if (event.getFace().equals(getPrimaryDrawableFace().getHumanFace())) {
                recomputeSymmetryPlane(0);
            } else {
                recomputeSymmetryPlane(1);
            }
        }    
    }
    
    private void recomputeSymmetryPlane(int index) {
        DrawableFace face = getCanvas().getScene().getDrawableFace(index);
        if (face == null) {
            return;
        }
        
        Logger log = Logger.measureTime();
        SymmetryEstimator estimator = new SymmetryEstimator(controlPanel.getSymmetryConfig());        
        face.getModel().compute(estimator);
        
        getCanvas().getScene().setDrawableSymmetryPlane(index,
                new DrawablePlane(estimator.getSymmetryPlaneMesh(), estimator.getSymmetryPlane()));
        getCanvas().getScene().setDrawableCuttingPlane(index,
                new DrawablePlane(estimator.getSymmetryPlaneMesh(), estimator.getSymmetryPlane()));
        getCanvas().getScene().setDrawableMirrorPlane(index,
                new DrawablePlane(estimator.getSymmetryPlaneMesh(), estimator.getSymmetryPlane()));

        getCanvas().getScene().hideSymmetryPlanes();
        log.printDuration("Symmetry plane estimation for " + face.getHumanFace().getShortName());

        face.getHumanFace().annouceEvent(new SymmetryPlaneComputed(face.getHumanFace(), face.getHumanFace().getShortName(), this));
    }
}
