package cz.fidentis.analyst.distance;

import cz.fidentis.analyst.core.ControlPanelBuilder;
import cz.fidentis.analyst.core.LoadedActionEvent;
import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.feature.FeaturePointType;
import cz.fidentis.analyst.scene.DrawableFpWeights;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.KeyStroke;

/**
 * @author Radek Oslejsek
 * @author Daniel Schramm
 */
public class FeaturePointsPanel extends JPanel {
    
    public static final String ACTION_COMMAND_FEATURE_POINT_HOVER_IN = "highlight hovered FP";
    public static final String ACTION_COMMAND_FEATURE_POINT_HOVER_OUT = "set default color of hovered FP";
    public static final String ACTION_COMMAND_FEATURE_POINT_SELECT = "highlight feature point with color";
    public static final String ACTION_COMMAND_FEATURE_POINT_RESIZE = "set size of feature point";
    public static final String ACTION_COMMAND_DISTANCE_RECOMPUTE = "recompute the Hausdorff distance";
    
    private Map<FeaturePointType, JLabel> featurePointStats;
    private List<JCheckBox> featurePointCheckBoxes;
    
    /**
     * Constructor.
     * 
     * @param action Action listener
     * @param featurePoints List of all feature points which can be used to calculate
     *                      the weighted Hausdorff distance
     * @param selectedFPs Set of feature point types which are initially selected
     *                    to be used to calculate the weighted Hausdorff distance
     */
    public void initComponents(ActionListener action, List<FeaturePoint> featurePoints, Set<FeaturePointType> selectedFPs) {
        final ControlPanelBuilder fpBuilder = new ControlPanelBuilder(this);
        
        featurePointStats = new HashMap<>(featurePoints.size());
        featurePointCheckBoxes = new ArrayList<>(featurePoints.size());
        
        for (int i = 0; i < featurePoints.size(); i++) {
            final FeaturePoint featurePoint = featurePoints.get(i);
            final FeaturePointType featurePointType = featurePoint.getFeaturePointType();
            
            final JCheckBox checkBox = fpBuilder.addCheckBox(selectedFPs.contains(featurePointType),
                    createListener(action, ACTION_COMMAND_FEATURE_POINT_SELECT, i)
            );
            checkBox.setText(featurePointType.getName());
            final int index = i;
            checkBox.addMouseListener(new MouseAdapter() {
                @Override
                public void mouseEntered(MouseEvent e) {
                    action.actionPerformed(new LoadedActionEvent(
                            e.getSource(),
                            ActionEvent.ACTION_PERFORMED,
                            ACTION_COMMAND_FEATURE_POINT_HOVER_IN,
                            index));
                }

                @Override
                public void mouseExited(MouseEvent e) {
                    action.actionPerformed(new LoadedActionEvent(
                            e.getSource(),
                            ActionEvent.ACTION_PERFORMED,
                            ACTION_COMMAND_FEATURE_POINT_HOVER_OUT,
                            index));
                }
            });
            featurePointCheckBoxes.add(checkBox);
            
            final JTextField sliderInput = fpBuilder.addSliderWithVal(100, null);
            sliderInput.setText(ControlPanelBuilder.doubleToStringLocale(DrawableFpWeights.FPW_DEFAULT_SIZE));
            sliderInput.postActionEvent(); // Set correct position of slider
            sliderInput.addActionListener(createListener(action, ACTION_COMMAND_FEATURE_POINT_RESIZE, i));
            // Modify listener of the ENTER key press
            final Object enterKeyAction = sliderInput.getInputMap()
                    .get(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0));
            final Action originalEnterAction = sliderInput.getActionMap()
                    .get(enterKeyAction);
            sliderInput.getActionMap().put(
                    enterKeyAction,
                    new AbstractAction() {
                        @Override
                        public void actionPerformed(ActionEvent e) {
                            originalEnterAction.actionPerformed(e);
                            if (!checkBox.isSelected()) {
                                return; // Recompute only if the feature point is selected
                            }
                            
                            action.actionPerformed(new ActionEvent(
                                    e.getSource(),
                                    ActionEvent.ACTION_PERFORMED,
                                    ACTION_COMMAND_DISTANCE_RECOMPUTE
                            ));
                        }
                    }
            );
            // Add mouse listener of the slider
            sliderInput.addMouseListener(new MouseAdapter() {
                private double oldValue = DrawableFpWeights.FPW_DEFAULT_SIZE;
                
                @Override
                public void mousePressed(MouseEvent e) {
                    oldValue = ControlPanelBuilder.parseLocaleDouble(sliderInput);
                }
                
                @Override
                public void mouseReleased(MouseEvent e) {
                    if (!checkBox.isSelected()
                            || oldValue == ControlPanelBuilder.parseLocaleDouble(sliderInput)) {
                        return; // Recompute only if the feature point is selected and value changed
                    }
                    
                    action.actionPerformed(new ActionEvent(
                            e.getSource(),
                            ActionEvent.ACTION_PERFORMED,
                            ACTION_COMMAND_DISTANCE_RECOMPUTE)
                    );
                }
            });
            
            fpBuilder.addGap();
            featurePointStats.put(featurePointType, fpBuilder.addLabelLine(null));
            fpBuilder.addGap();
            
            fpBuilder.addLine();
        }
        
        /*
        fpBuilder.addButton("Select all", (ActionEvent ae) -> {
            featurePointCheckBoxes.forEach(fpCheckBox ->
                    fpCheckBox.setSelected(true)
            );
            action.actionPerformed(new ActionEvent(
                    ae.getSource(),
                    ActionEvent.ACTION_PERFORMED,
                    ACTION_COMMAND_FEATURE_POINT_SELECT_ALL)
            );
        });
        fpBuilder.addButton("Deselect all", (ActionEvent ae) -> {
            featurePointCheckBoxes.forEach(fpCheckBox ->
                    fpCheckBox.setSelected(false)
            );
            action.actionPerformed(new ActionEvent(
                    ae.getSource(),
                    ActionEvent.ACTION_PERFORMED,
                    ACTION_COMMAND_FEATURE_POINT_SELECT_NONE)
            );
        });
        fpBuilder.addLine();
        */
        
    }
    
    /**
     * Updates GUI elements that display the weights of feature points
     * used to calculate the weighted Hausdorff distance.
     * 
     * @param featurePointWeights Map of feature point types and their weights
     */
    public void updateFeaturePointWeights(Map<FeaturePointType, Double> featurePointWeights) {
        featurePointStats.forEach((fpType, fpWeightLabel) -> {
            final Double fpWeight = featurePointWeights.get(fpType);
            fpWeightLabel.setText(fpWeight == null ? "" : String.format("%.3f", fpWeight));
            fpWeightLabel.setFont(new Font("Arial", 1, 12));
        });
    }
    
    /**
     * (De)selects all feature points for the computation of the weighted Hausdorff distance.
     * 
     * @param selected {@code true} if all feature point checkboxes are to be selected,
     *                 {@code false} otherwise
     */
    public void selectAllFeaturePoints(boolean selected) {
        featurePointCheckBoxes.forEach(fpCheckBox ->
            fpCheckBox.setSelected(selected)
        );
    }
    
    /**
     * (De)selects given feature point.
     * 
     * @param index Index of the feature point
     * @param selected {@code true} if all feature point checkboxes are to be selected,
     *                 {@code false} otherwise
     */
    public void selectFeaturePoint(int index, boolean selected) {
        if (index >= 0 && index < featurePointCheckBoxes.size()) {
            featurePointCheckBoxes.get(index).setSelected(selected);
        }
    }
    
    /**
     * Creates and returns action listener that can be connected with a low-level 
     * GUI element (e.g., a button). Action event of the low-level element is then
     * re-directed to the given {@code ControlPanelAction} as given command.
     * The listener may also carry additional data as a payload.
     * 
     * @param action An instance of the {@link ControlPanelAction}
     * @param command Control panel command
     * @param data Payload data of the action listener
     * @return Action listener
     */
    protected final ActionListener createListener(ActionListener action, String command, Object data) {
        return new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                action.actionPerformed(new LoadedActionEvent(
                        e.getSource(), 
                        ActionEvent.ACTION_PERFORMED, 
                        command,
                        data)
                ); 
            }  
        };
    }
}
