package cz.fidentis.analyst.core;

import java.awt.event.ActionEvent;
import java.text.NumberFormat;
import javax.swing.event.ChangeEvent;
import javax.swing.text.DefaultFormatterFactory;
import javax.swing.text.NumberFormatter;

/**
 * ComboSlider with integer numbers.
 * 
 * @author Radek Oslejsek
 */
public class ComboSliderInteger extends ComboSlider {
    
    private int min = 0;
    private int max = 100;
    
    /**
     * Constructor. The default scale is from 0 to 100.
     */
    public ComboSliderInteger() {
        setRange(0, 100);
        connect();
    }
    
    /**
     * Sets the slider and input field range.
     * 
     * @param min Minimum value
     * @param max Maximum value. If less than {@code min}, then {@code min} is used instead.
     */
    public final void setRange(int min, int max) {
        this.min = min;
        this.max = (max < min) ? min : max;
        
        NumberFormatter formatter = new NumberFormatter(NumberFormat.getInstance());
        formatter.setValueClass(Integer.class);
        formatter.setMinimum(this.min);
        formatter.setMaximum(this.max);
        formatter.setAllowsInvalid(false);
        
        getInputField().setFormatterFactory(new DefaultFormatterFactory(formatter));
        getInputField().setText(Integer.toString(this.min));
        
        getSlider().setMinimum(this.min);
        getSlider().setMaximum(this.max);
        
        setValue(this.min);
        getInputField().postActionEvent(); // invoke textField action listener
    }
    
    /**
     * Returns the value.
     * 
     * @return the value.
     */
    public int getValue() {
        return TextFieldUtils.parseLocaleInt(getInputField());
    }

    /**
     * Sets the value of the slider and input field. 
     * If the value is outside of the range then nothing happens.
     * 
     * @param value a new value
     */
    public void setValue(int value) {
        if (value >= getMinimum() && value <= getMaximum()) {
            getInputField().setText(TextFieldUtils.intToStringLocale(value));
            getInputField().postActionEvent(); // invoke textField action listener
        }
    }
    
    /**
     * Returns the lower limit of the range.
     * 
     * @return the lower limit
    */
    public int getMinimum() {
        return this.min;
    }

    /**
     * Returns the upper limit of the range.
     * 
     * @return the upper limit
     */
    public int getMaximum() {
        return this.max;
    }
    
    private void connect() {
        // update input field on slider's change and inform external listeners
        getSlider().addChangeListener((ChangeEvent ce) -> { 
            setValue(getSlider().getValue()); // also triggers the event
        });
        
        // update slider when the input field changed
        getInputField().addActionListener((ActionEvent ae) -> { 
            getSlider().setValue(TextFieldUtils.parseLocaleInt(getInputField()));
        });
    }
}
