package cz.fidentis.analyst.core;

import cz.fidentis.analyst.batch.BatchAction;
import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.canvas.toolbar.SceneToolboxFaceToFace;
import cz.fidentis.analyst.canvas.toolbar.SceneToolboxSingleFace;
import cz.fidentis.analyst.curvature.CurvatureAction;
import cz.fidentis.analyst.distance.DistanceAction;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.registration.RegistrationAction;
import cz.fidentis.analyst.symmetry.ProfilesAction;
import cz.fidentis.analyst.symmetry.SymmetryAction;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.nio.file.Path;
import java.util.List;
import java.util.Objects;
import javax.swing.GroupLayout;
import javax.swing.JScrollPane;
import javax.swing.LayoutStyle;
import org.openide.windows.TopComponent;

/**
 * The non-singleton window/tab for the analysis of one, two or many to many faces
 *
 * @author Matej Kovar
 */
public class FaceTab extends TopComponent {
    
    private final Canvas canvas ;
    private final TopControlPanel controlPanel;
    private final JScrollPane scrollPane;
    
    private String nameOfFace1 = null;
    private String nameOfFace2 = null;
    private ActionListener listener = null;
    
    /**
     * Constructor for single face analysis and 1:1.
     * 
     * @param primary Primary face
     * @param secondary Secondary face
     * @param name Tab name
     * @param listener action listener
     */
    public FaceTab(HumanFace primary, HumanFace secondary, String name, ActionListener listener) {
        canvas = new Canvas();
        this.listener = listener;
        
        nameOfFace1 = primary.getShortName();

        canvas.addPrimaryFace(primary);

        if (secondary == null) { // single face analysis
            canvas.getScene().setDefaultColors();
            canvas.addToolBox(new SceneToolboxSingleFace(canvas));
            nameOfFace2 = null;
        } else { // 1:1
            canvas.addSecondaryFace(secondary);
            canvas.getScene().setDefaultColors();
            canvas.addToolBox(new SceneToolboxFaceToFace(canvas));
            nameOfFace2 = secondary.getShortName();
        }

        controlPanel = new TopControlPanel();
        scrollPane = new JScrollPane(controlPanel);
        
        setName(name);
        initComponents();
        
        // change the height so that it corresponds to the height of the OpenGL window
        canvas.addComponentListener(new ComponentAdapter() {
            @Override
            public void componentResized(ComponentEvent e) {
                scrollPane.setSize(ControlPanel.CONTROL_PANEL_WIDTH, canvas.getHeight());
            }
        });
        
        controlPanel.addChangeListener(e -> getCanvas().renderScene());
        
        if (secondary == null) { // single face analysis
            new CurvatureAction(getCanvas(), controlPanel);
            new SymmetryAction(getCanvas(), controlPanel);
            new ProfilesAction(getCanvas(), controlPanel);
        } else { // 1:1
            new RegistrationAction(canvas, controlPanel);
            new DistanceAction(canvas, controlPanel);
            new SymmetryAction(canvas, controlPanel);
            new ProfilesAction(canvas, controlPanel);
        }
    }
    
    /**
     * Constructor for N:N
     * 
     * @param faces Paths to faces
     * @param name Tab name
     * @param listener action listener
     */
    public FaceTab(List<Path> faces, String name, ActionListener listener) {
        canvas = new Canvas();
        this.listener = listener;
        
        canvas.addToolBox(new SceneToolboxSingleFace(canvas));

        controlPanel = new TopControlPanel();
        scrollPane = new JScrollPane(controlPanel);
        
        setName(name);
        initComponents();
        
        // change the height so that it corresponds to the height of the OpenGL window
        canvas.addComponentListener(new ComponentAdapter() {
            @Override
            public void componentResized(ComponentEvent e) {
                scrollPane.setSize(ControlPanel.CONTROL_PANEL_WIDTH, canvas.getHeight());
            }
        });
        
        controlPanel.addChangeListener(e -> getCanvas().renderScene());
        
        new BatchAction(canvas, controlPanel, faces);
    }
    
    @Override
    public int getPersistenceType() {
        return TopComponent.PERSISTENCE_NEVER; // TO DO: change to .PERSISTENCE_ONLY_OPENED when we can re-create the ProjectTC
    }
    
    private void initComponents() {
        GroupLayout layout = new GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
                layout.createParallelGroup(GroupLayout.Alignment.LEADING)
                        .addGroup(layout.createSequentialGroup()
                                .addPreferredGap(LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(canvas, GroupLayout.DEFAULT_SIZE, 651, Short.MAX_VALUE)
                                .addPreferredGap(LayoutStyle.ComponentPlacement.RELATED)
//                                .addComponent(renderingToolBar, GroupLayout.PREFERRED_SIZE, RenderingToolBar.WIDTH, GroupLayout.PREFERRED_SIZE)
                                .addComponent(
                                        scrollPane,
                                        ControlPanel.CONTROL_PANEL_WIDTH, 
                                        ControlPanel.CONTROL_PANEL_WIDTH, 
                                        ControlPanel.CONTROL_PANEL_WIDTH
                                )
                        )
        );
        layout.setVerticalGroup(
                layout.createParallelGroup(GroupLayout.Alignment.LEADING)
                        .addGroup(layout.createSequentialGroup()
                                .addGroup(layout.createBaselineGroup(true, true)
                                        .addComponent(canvas)
//                                        .addComponent(renderingToolBar)
                                        .addComponent(scrollPane)
                                ))
        );
    }

    public String getNameOfFace1() {
        return nameOfFace1;
    }

    public String getNameOfFace2() {
        return nameOfFace2;
    }
    
    /**
     * Checks whether this tab contains name of face
     * @param name String name of face
     * @return true if face with this name is in this tab
     */
    public boolean hasFace(String name) {
        return (name.equals(nameOfFace1) || name.equals(nameOfFace2));
    }

    public Canvas getCanvas() {
        return canvas;
    }
    
    
    @Override
    public boolean canClose() {
        if (nameOfFace1 != null && nameOfFace2 == null) {
            listener.actionPerformed(new ActionEvent(this, ActionEvent.ACTION_PERFORMED, "FaceTab"));
        } else if (nameOfFace1 != null && nameOfFace2 != null) {
            listener.actionPerformed(new ActionEvent(this, ActionEvent.ACTION_PERFORMED, "FaceToFaceTab"));
        } else {
            listener.actionPerformed(new ActionEvent(this, ActionEvent.ACTION_PERFORMED, "ManyToManyTab"));
        }
        return super.canClose(); //To change body of generated methods, choose Tools | Templates.
    }

    
    @Override
    public int hashCode() {
        int hash = 5;
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final FaceTab other = (FaceTab) obj;
        if (!Objects.equals(this.nameOfFace1, other.nameOfFace1)) {
            return false;
        }
        return Objects.equals(this.nameOfFace2, other.nameOfFace2);
    }
    
    
}