package cz.fidentis.analyst.symmetry;

import static cz.fidentis.analyst.gui.UserInterface.frameMain;
import cz.fidentis.analyst.mesh.core.CornerTableRow;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.swing.ImageIcon;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.ProgressMonitor;
import javax.swing.UIManager;
import javax.vecmath.Vector3d;

/**
 *
 * @author Natália Bebjaková
 * 
 * Main class for computing approximate plane of symmetry of the 3D model. 
 * Default values of the configuration are given due to the best results on tested objects.
 * On many different 3D models, it exists other values of config that will have better impact on result. 
 * 
 */
public class SymmetryCounter {
    /**
     * Facet of the model on which symmetry is computed 
     */
    private final MeshFacet facet;   
    /**
     * Representation for areas of Voronoi region of triangles
     */
    private TriangleVertexAreas[] areas;
    /**
     * Represent min-max box. It is automatically maintained by given point array.
     */
    private BoundingBox boundingBox;
    /**
     * Helping array of triangles computed from corner table 
     */
    protected Triangle[] triangles;
    /**
     * panel for configuration of symmetry counting 
     */
    private JPanel panel;

    /**
     * 
     * @return panel for configuration of symmetry counting 
     */
    public JPanel getPanel() {
        return panel;
    }

    /**
     * 
     * @param panel new panel for configuration of symmetry counting 
     */
    public void setPanel(JPanel panel) {
        this.panel = panel;
    }
    

    /**
     * 
     * @return Facet of the model on which symmetry is computed 
     */
    public MeshFacet getFacet() {
        return facet;
    }

    /**
     * Creates new class for computing symmetry
     * 
     * @param f facet on which symmetry will be computed 
     */
    public SymmetryCounter(MeshFacet f) {
        this.facet = f;
    }
    
    /**
     * If bounding box is not created yet, it creates new one. 
     * 
     * @return Represent min-max box of the boundries of the model. 
     */
    public BoundingBox getBoundingBox() {
        if (boundingBox == null) {
            boundingBox = new BoundingBox(facet.getVertices());               
        }
        return boundingBox; 
    }

    /**
     * 
     * @param boundingBox new min-max box of the boundries of the model. 
     */
    public void setBoundingBox(BoundingBox boundingBox) {
        this.boundingBox = boundingBox;
    }
    
    /**
     * 
     * @return array of triangles 
     */
    public Triangle[] getTriangles() {
        return triangles;
    }
    
    /**
     * 
     * @param triangles new array of triangles 
     */
    public void setTriangles(Triangle[] triangles) {
        this.triangles = triangles;
    }
    
    /**
     * Representation of areas of Voronoi region of triangles
     * used for computing Gaussian curvature 
     */
    private class TriangleVertexAreas {
            public double v1Area;
            public double v2Area;
            public double v3Area;
    }
   
    /**
     * Initialize values necessary for computing
     */
    public void init() {
        initTriangles();
        initArrayOfTriangleVertexAreas();
        boundingBox = new BoundingBox(facet.getVertices());
    }
    
    /**
     * Computes triangles of facet from corner table
     */
    public void initTriangles() {
        triangles = new Triangle[facet.getCornerTable().getSize() / 3];
        for (int i = 0; i < facet.getCornerTable().getSize(); i += 3) {
            Triangle t = new Triangle(facet.getCornerTable().getRow(i).getVertexIndex(),
                    facet.getCornerTable().getRow(i + 1).getVertexIndex(),
                    facet.getCornerTable().getRow(i + 2).getVertexIndex());
            triangles[(i / 3)] = t;
        }     
    }
    
    /**
     * Calculates new normals of the points
     * 
     * @return new normals represented as mesh points so that math operations can be done 
     */
    public MeshPoint[] CalculateNormals() {
        MeshPoint[] newNormals = new MeshPoint[facet.getNumberOfVertices()];
        for (int i = 0; i < facet.getNumberOfVertices(); i++)
            newNormals[i] = new MeshPoint(new Vector3d(0, 0, 0), null, null);
        for (Triangle t : triangles) {
            MeshPoint triangleNormal = (facet.getVertices().get(t.vertex3).subtractPosition
        (facet.getVertices().get(t.vertex1))).crossProduct(facet.getVertices().get(t.vertex2).subtractPosition(facet.getVertices().get(t.vertex1)));
            newNormals[t.vertex1] = newNormals[t.vertex1].addPosition(new MeshPoint(triangleNormal.getPosition(), null, null));
            newNormals[t.vertex2] = newNormals[t.vertex2].addPosition(new MeshPoint(triangleNormal.getPosition(), null, null));
            newNormals[t.vertex3] = newNormals[t.vertex3].addPosition(new MeshPoint(triangleNormal.getPosition(), null, null));
        }
        for (int i = 0; i < newNormals.length; i++){ 
            if (newNormals[i].getPosition().x != 0 && newNormals[i].getPosition().y != 0 && newNormals[i].getPosition().z != 0) {
                newNormals[i] = newNormals[i].dividePosition(newNormals[i].abs());
            }   
        }
        return newNormals;
    }
    
    /**
     * 
     * @param pointIndex index of point for which we are searching traingles in its neighborhood
     * @return triangle neighbours of given index of vertex
     */
    public List<Integer> getNeighbours(int pointIndex) {
        return facet.getCornerTable().getTriangleIndexesByVertexIndex(pointIndex);
    }
    
    /**
     * Return area of Voronoi region of triangle
     * 
     * @param t triangle of which area is computed
     * @return area of Voronoi region
     */
    private TriangleVertexAreas computeTriangleVertexAreas(Triangle t) {
        MeshPoint circumcenter;
                
        MeshPoint tVertex1 = facet.getVertices().get(t.vertex1);
        MeshPoint tVertex2 = facet.getVertices().get(t.vertex2);
        MeshPoint tVertex3 = facet.getVertices().get(t.vertex3);
        
        double a = (tVertex2.subtractPosition(tVertex3)).abs();
        double b = (tVertex3.subtractPosition(tVertex1)).abs();
        double c = (tVertex2.subtractPosition(tVertex1)).abs();
        
        double d1 = a * a * (b * b + c * c - a * a);
        double d2 = b * b * (c * c + a * a - b * b);
        double d3 = c * c * (a * a + b * b - c * c);
        double dSum = d1 + d2 + d3;
        
        d1 /= dSum;
        d2 /= dSum;
        d3 /= dSum;

        MeshPoint v1Half = (tVertex2.addPosition(tVertex3)).dividePosition(2);
        MeshPoint v2Half = (tVertex1.addPosition(tVertex3)).dividePosition(2);
        MeshPoint v3Half = (tVertex2.addPosition(tVertex1)).dividePosition(2);

        
        TriangleVertexAreas area = new TriangleVertexAreas();

        if (d1 < 0) {           
            area.v3Area = ((v2Half.subtractPosition(tVertex3)).crossProduct(v1Half.subtractPosition(tVertex3))).abs() / 2;
            area.v2Area = ((v3Half.subtractPosition(tVertex2)).crossProduct(v1Half.subtractPosition(tVertex2))).abs() / 2;
            area.v1Area = (((v1Half.subtractPosition(tVertex1)).crossProduct(v3Half.subtractPosition(tVertex1))).abs() / 2) + 
                    (((v1Half.subtractPosition(tVertex1)).crossProduct(v2Half.subtractPosition(tVertex1))).abs() / 2);
            return area;
        }
        if (d2 < 0) {
            area.v3Area = ((v3Half.subtractPosition(tVertex1)).crossProduct(v2Half.subtractPosition(tVertex1))).abs() / 2;
            area.v2Area = ((v1Half.subtractPosition(tVertex3)).crossProduct(v2Half.subtractPosition(tVertex3))).abs() / 2;
            area.v1Area = (((v2Half.subtractPosition(tVertex2)).crossProduct(v1Half.subtractPosition(tVertex2))).abs() / 2) + 
                    (((v2Half.subtractPosition(tVertex2)).crossProduct(v3Half.subtractPosition(tVertex2))).abs() / 2);
            return area;
        }
        if (d3 < 0) {
            area.v2Area = ((v1Half.subtractPosition(tVertex2)).crossProduct(v3Half.subtractPosition(tVertex2))).abs() / 2;
            area.v1Area = ((v2Half.subtractPosition(tVertex1)).crossProduct(v3Half.subtractPosition(tVertex1))).abs() / 2;
            area.v3Area = (((v3Half.subtractPosition(tVertex3)).crossProduct(v2Half.subtractPosition(tVertex3))).abs() / 2) + 
                    (((v3Half.subtractPosition(tVertex3)).crossProduct(v1Half.subtractPosition(tVertex3))).abs() / 2);
            return area;
        }
        
        circumcenter = tVertex1.multiplyPosition(d1).addPosition(tVertex2.multiplyPosition(d2).addPosition(tVertex3.multiplyPosition(d3)));
        
        area.v1Area = (((v2Half.subtractPosition(tVertex1)).crossProduct(circumcenter.subtractPosition(tVertex1))).abs() / 2) + 
                (((v3Half.subtractPosition(tVertex1)).crossProduct(circumcenter.subtractPosition(tVertex1))).abs() / 2);
        
        area.v2Area = (((v3Half.subtractPosition(tVertex2)).crossProduct(circumcenter.subtractPosition(tVertex2))).abs() / 2) +
                (((v1Half.subtractPosition(tVertex2)).crossProduct(circumcenter.subtractPosition(tVertex2))).abs() / 2);
        
        area.v3Area = (((v1Half.subtractPosition(tVertex3)).crossProduct(circumcenter.subtractPosition(tVertex3))).abs() / 2) +
                (((v2Half.subtractPosition(tVertex3)).crossProduct(circumcenter.subtractPosition(tVertex3))).abs() / 2);
        return area;
    }
    
    /**
     * Computes area of Voronoi region for all triangles
     */
    private void initArrayOfTriangleVertexAreas() {
        areas = new TriangleVertexAreas[triangles.length];
        for (int i = 0; i < areas.length; i++) {
            areas[i] = computeTriangleVertexAreas(triangles[i]);
        }
    }
 
    /**
     * Calculates angle of the triangle according to centerIndex
     * 
     * @param centerIndex centerIndex
     * @param t triangle
     * @return angle
     */
    private double calculateTriangleAlfa(int centerIndex, Triangle t) {        
        MeshPoint tVertex1 = facet.getVertices().get(t.vertex1);
        MeshPoint tVertex2 = facet.getVertices().get(t.vertex2);
        MeshPoint tVertex3 = facet.getVertices().get(t.vertex3);
        
        double alfaCos = 0;
        
        if (t.vertex1 == centerIndex) {
            MeshPoint e1 = tVertex1.subtractPosition(tVertex3);
            e1 = e1.dividePosition(e1.abs());
            
            MeshPoint e2 = tVertex2.subtractPosition(tVertex3);
            e2 = e2.dividePosition(e2.abs());
            
            alfaCos = e1.dotProduct(e2);
        }
        else if (t.vertex2 == centerIndex) {
            
            MeshPoint e1 = tVertex2.subtractPosition(tVertex1);
            e1 = e1.dividePosition(e1.abs());
            
            MeshPoint e2 = tVertex3.subtractPosition(tVertex1);
            e2 = e2.dividePosition(e2.abs());
            
            alfaCos = e1.dotProduct(e2);
            }
        else if (t.vertex3 == centerIndex){
            MeshPoint e1 = tVertex3.subtractPosition(tVertex2);
            e1 = e1.dividePosition(e1.abs());
                
            MeshPoint e2 = tVertex1.subtractPosition(tVertex2);
            e2 = e2.dividePosition(e2.abs());
            
            alfaCos = e1.dotProduct(e2);
            }
        double alfa = Math.acos(alfaCos);
        return 1 / Math.tan(alfa);
    }

    /**
     * Calculates angle of the triangle according to centerIndex
     * 
     * @param centerIndex centerIndex
     * @param t triangle
     * @return angle
     */
    private double calculateTriangleBeta(int centerIndex, Triangle t) {
        MeshPoint tVertex1 = facet.getVertices().get(t.vertex1);
        MeshPoint tVertex2 = facet.getVertices().get(t.vertex2);
        MeshPoint tVertex3 = facet.getVertices().get(t.vertex3);
        
        double betaCos = 0;
            
        if (t.vertex1 == centerIndex) {
            MeshPoint e1 = tVertex1.subtractPosition(tVertex2);
            e1 = e1.dividePosition(e1.abs());
            
            MeshPoint e2 = tVertex3.subtractPosition(tVertex2);
            e2 = e2.dividePosition(e2.abs());
            
            betaCos = e1.dotProduct(e2);
        }
        else if (t.vertex2 == centerIndex) {
            
            MeshPoint e1 = tVertex2.subtractPosition(tVertex3);
            e1 = e1.dividePosition(e1.abs());
            
            MeshPoint e2 = tVertex1.subtractPosition(tVertex3);
            e2 = e2.dividePosition(e2.abs());
            
            betaCos = e1.dotProduct(e2);
            }
        else if (t.vertex3 == centerIndex){
            MeshPoint e1 = tVertex3.subtractPosition(tVertex1);
            e1 = e1.dividePosition(e1.abs());
                
            MeshPoint e2 = tVertex2.subtractPosition(tVertex1);
            e2 = e2.dividePosition(e2.abs());
            
            betaCos = e1.dotProduct(e2);
            }
            double beta = Math.acos(betaCos);
            return 1 / Math.tan(beta);
        }
        
    /**
     * Calculates Laplacian 
     * 
     * @param centerIndex centerIndex
     * @return laplacian 
     */
    private MeshPoint calculateLaplacian(int centerIndex) {
        List<Integer> trianglesNeighbours = getNeighbours(centerIndex);

        if (trianglesNeighbours.isEmpty()) {
            return new MeshPoint(new Vector3d(), new Vector3d(), new Vector3d());
        }
        double areaSum = 0;
        MeshPoint pointSum = new MeshPoint((new Vector3d(0.0, 0.0, 0.0)), null, null);
        for (int i = 0; i < trianglesNeighbours.size(); i++) {
            Triangle t = triangles[trianglesNeighbours.get(i)];
            Triangle tNext = triangles[trianglesNeighbours.get((i + 1) % trianglesNeighbours.size())];
            double area = 0;
            
            MeshPoint tVertex1 = facet.getVertices().get(t.vertex1);
            MeshPoint tVertex2 = facet.getVertices().get(t.vertex2);
            MeshPoint tVertex3 = facet.getVertices().get(t.vertex3);
            MeshPoint centerPoint = facet.getVertices().get(centerIndex);
                    
            if (t.vertex1 == centerIndex) {
                area = areas[trianglesNeighbours.get(i)].v1Area;
                pointSum = pointSum.addPosition((tVertex2.subtractPosition(centerPoint)).multiplyPosition((calculateTriangleAlfa(centerIndex, t)
                        + calculateTriangleBeta(centerIndex, tNext)) / 2));
                       
            }
            else if (t.vertex2 == centerIndex) {
                area = areas[trianglesNeighbours.get(i)].v2Area;
                pointSum = pointSum.addPosition((tVertex3.subtractPosition(centerPoint)).multiplyPosition((calculateTriangleAlfa(centerIndex, t)
                        + calculateTriangleBeta(centerIndex, tNext)) / 2));
            }
            else if (t.vertex3 == centerIndex) {
                area = areas[trianglesNeighbours.get(i)].v3Area;
                pointSum = pointSum.addPosition((tVertex1.subtractPosition(centerPoint)).multiplyPosition((calculateTriangleAlfa(centerIndex, t)
                        + calculateTriangleBeta(centerIndex, tNext)) / 2));
            }
            areaSum += area;
        }
        return pointSum.multiplyPosition(1 / areaSum);
    }
    
    /**
     * Gaussian curvature in a vertex of a triangle mesh.
     * It can only be estimated beacuse triangle mesh is not a continuous
     * but discrete representation of surface.
     * 
     * @param centerIndex index of vertex in which gaussian curvature is computed
     * @return Gaussian curvature in given vertex
     */
    private double getGaussianCurvature(int centerIndex) {
        
        List<Integer> triangleNeighbours = this.getNeighbours(centerIndex);
        
        if (triangleNeighbours.isEmpty()) {
            return Double.NaN;
        }
        
        double sum = 2 * Math.PI;
        double areaSum = 0;

        for (int i = 0; i < triangleNeighbours.size(); i++) {
            MeshPoint v1 = new MeshPoint(new Vector3d(), null, null);
            MeshPoint v2 = new MeshPoint(new Vector3d(), null, null);
            Triangle t = triangles[triangleNeighbours.get(i)];
            
            MeshPoint tVertex1 = facet.getVertices().get(t.vertex1);
            MeshPoint tVertex2 = facet.getVertices().get(t.vertex2);
            MeshPoint tVertex3 = facet.getVertices().get(t.vertex3);
            
            double area = 0;
            if (t.vertex1 == centerIndex) {
                v1 = tVertex2.subtractPosition(tVertex1);
                v2 = tVertex3.subtractPosition(tVertex1);

                area = areas[triangleNeighbours.get(i)].v1Area;
            }
            else if (t.vertex2 == centerIndex) {
                v1 = tVertex1.subtractPosition(tVertex2);
                v2 = tVertex3.subtractPosition(tVertex2);
                
                area = areas[triangleNeighbours.get(i)].v2Area;
            }
            else if (t.vertex3 == centerIndex) {
                v1 = tVertex2.subtractPosition(tVertex3);
                v2 = tVertex1.subtractPosition(tVertex3);

                area = areas[triangleNeighbours.get(i)].v3Area;
            }
            
            areaSum += area;
            v1 = v1.dividePosition(v1.abs());
            v2 = v2.dividePosition(v2.abs());

            sum -= Math.acos(v1.dotProduct(v2));
        } 
        return sum * (1.0 / areaSum);
       
    }

    /**
     * 
     * @param centerIndex center index
     * @return mean curvature
     */
    private double getMeanCurvature(int centerIndex) {
        return calculateLaplacian(centerIndex).abs();
    }
    
    /**
     * 
     * @param centerIndex center index
     * @return max curvature
     */
    private double getMaxCurvature(int centerIndex) {
        double gaussianCurvature = getGaussianCurvature(centerIndex);
        if (Double.isNaN(gaussianCurvature)) {
            return Double.NaN;
        }
        double meanCurvature = getMeanCurvature(centerIndex);
        double meanCurvatureSqr = meanCurvature * meanCurvature;
        if (meanCurvatureSqr <= gaussianCurvature) { 
            return meanCurvature;
        }
        return meanCurvature + Math.sqrt(meanCurvatureSqr - gaussianCurvature);
    }
    
    /**
     * Computes votes for given plane 
     * 
     * @param plane Plane for which votes are computed
     * @param curvatures significant curvatures chosen for computing
     * @param points significant point chosen for computing
     * @param minCurvRatio optional parameter from configuration
     * @param minAngleCos optional parameter from configuration
     * @param minNormAngleCos optional parameter from configuration
     * @param maxDist optional parameter from configuration
     * @return total votes given to plane while computing the symmetry
     */
    private int GetVotes(Plane plane,
            ArrayList<Double> curvatures,
            ArrayList<Integer> points,
            double minCurvRatio,
            double minAngleCos,
            double minNormAngleCos,
            double maxDist) {
        
        plane.normalize();
        
        MeshPoint normal = new MeshPoint((new Vector3d(plane.a, plane.b, plane.c)),null, null);
        double d = plane.d;
        double maxCurvRatio = 1.0 / minCurvRatio;
        int votes = 0;
        MeshPoint[] normals = CalculateNormals();

        for (int i = 0; i < curvatures.size(); i++) {
            for (int j = 0; j < curvatures.size(); j++) {
                if (i != j && (curvatures.get(i) / curvatures.get(j) >= minCurvRatio
                        && curvatures.get(i) / curvatures.get(j) <= maxCurvRatio)) {
                    
                    MeshPoint vec = facet.getVertices().get(points.get(i)).subtractPosition(facet.getVertices().get(points.get(j)));
                    vec = vec.dividePosition(vec.abs());
                    double cos = vec.dotProduct(normal);
 
                    MeshPoint ni;
                    
                    /*if (facet.getVertices().get(points.get(i)).getNormal() != null) {
                        ni = new MeshPoint(new Vector3d(facet.getVertices().get(points.get(i)).getNormal().x,
                            facet.getVertices().get(points.get(i)).getNormal().y,
                            facet.getVertices().get(points.get(i)).getNormal().z), null, null);
                    } else {
                        ni = new MeshPoint (new Vector3d(normals[points.get(i)].getPosition().x,normals[points.get(i)].getPosition().y,
                           normals[points.get(i)].getPosition().z), null, null);
                    }
                    
                    ni = ni.divide(ni.abs());
                    
                    MeshPoint nj;
                    if (facet.getVertices().get(points.get(j)).getNormal() != null) {
                        nj = new MeshPoint(new Vector3d(facet.getVertices().get(points.get(j)).getNormal().x,
                            facet.getVertices().get(points.get(j)).getNormal().y,
                            facet.getVertices().get(points.get(j)).getNormal().z), null, null); 
                    } else {
                        nj = new MeshPoint (new Vector3d(normals[points.get(j)].getPosition().x,normals[points.get(j)].getPosition().y,
                               normals[points.get(j)].getPosition().z), null, null);
                    }*/
                   
                        
                    ni = new MeshPoint (new Vector3d(normals[points.get(i)].getPosition().x,normals[points.get(i)].getPosition().y,
                           normals[points.get(i)].getPosition().z), null, null);
                        
                    ni = ni.dividePosition(ni.abs());
                        
                    MeshPoint nj;
                    nj = new MeshPoint (new Vector3d(normals[points.get(j)].getPosition().x,normals[points.get(j)].getPosition().y,
                               normals[points.get(j)].getPosition().z), null, null);
                    nj = nj.dividePosition(nj.abs());

                    MeshPoint normVec = ni.subtractPosition(nj);
                    normVec = normVec.dividePosition(normVec.abs());
                    double normCos = normVec.dotProduct(normal);
                    
                    if (Math.abs(cos) >= minAngleCos && Math.abs(normCos) >= minNormAngleCos) {
                        MeshPoint avrg = (facet.getVertices().get(points.get(i)).addPosition(facet.getVertices().get(points.get(j)))).dividePosition(2.0);

                        double dist = normal.getPosition().x * avrg.getPosition().x +
                                normal.getPosition().y * avrg.getPosition().y + normal.getPosition().z * avrg.getPosition().z + d;
                        
                        dist = Math.abs(dist);
                        
                        if (dist <= maxDist) {                 
                            votes++;  
                        }   
                    }
                }
            }
        }
        return votes;
    }
    
    /**
     * 
     * @param conf Configuration for computing
     * @return approximate plane of symmtetry
     */
    public Plane getAproxSymmetryPlane(Config conf) {
        UIManager.put("ProgressMonitor.progressText", "Counting symmetry...");
 
        ArrayList<AproxSymmetryPlane> planes = new ArrayList<>();
        MeshPoint[] normals = CalculateNormals();
        double[] curvatures = new double[facet.getNumberOfVertices()];
        for (int i = 0; i < facet.getNumberOfVertices(); i++) {
            if (facet.getNumberOfVertices() > 2000) {
                curvatures[i] = this.getMaxCurvature(i);
            } else {
                curvatures[i] = this.getGaussianCurvature(i);
            }
            if (Double.isNaN(curvatures[i])){
                curvatures[i] = Double.MIN_VALUE;
            }
        }
        ArrayList<Double> sortedCurvatures = new ArrayList<>();
        for (int i = 0; i < curvatures.length; i++) {
            sortedCurvatures.add(curvatures[i]);
        }
        Collections.sort(sortedCurvatures);
           
        if(conf.getSignificantPointCount() > facet.getNumberOfVertices()) {
            conf.setSignificantPointCount((facet.getNumberOfVertices()) - 1);
        }
        double bottomCurvature = sortedCurvatures.get(sortedCurvatures.size() - 1 - conf.getSignificantPointCount());   
      
        ArrayList<Integer> significantPoints = new ArrayList<>();
        ArrayList<Double> significantCurvatures = new ArrayList<>();
        
        for (int i = 0; i < facet.getNumberOfVertices(); i++) {
            if (curvatures[i] >= bottomCurvature) {
                significantCurvatures.add(curvatures[i]);
                significantPoints.add(i);
            }
        }
        
        Plane plane = new Plane(0, 0, 0, 0);
        int lastVotes = 0;
        
        ProgressMonitor progressMonitor;
        progressMonitor = new ProgressMonitor(panel, "Cunting...",
                "Steps", 0, significantPoints.size());
        
        double onePercent = significantCurvatures.size() / 100.0;
        double percentsPerStep = 1 / onePercent;
        for (int i = 0; i < significantPoints.size(); i++) {
            for (int j = 0; j < significantPoints.size(); j++) {
                if (i != j) {
                    double minRatio = conf.getMinCurvRatio();
                    double maxRatio = 1.0 / minRatio;
                    if (significantCurvatures.get(i) / significantCurvatures.get(j) >= minRatio && significantCurvatures.get(i) /
                            significantCurvatures.get(j) <= maxRatio) {
                        
                        MeshPoint p1 = facet.getVertex(significantPoints.get(i));
                        MeshPoint p2 = facet.getVertex(significantPoints.get(j));
                        MeshPoint avrg = (p1.addPosition(p2)).dividePosition(2.0);
                        MeshPoint normal = (p1.subtractPosition(p2));
                        
                        normal = normal.dividePosition(normal.abs());
                        double d = -(normal.getPosition().x * avrg.getPosition().x) -
                                (normal.getPosition().y * avrg.getPosition().y) - (normal.getPosition().z * avrg.getPosition().z);
                              
                        MeshPoint ni;
                        /*if (facet.getVertex(significantPoints.get(i)).getNormal() != null) {
                            ni = new MeshPoint(new Vector3d(facet.getVertex(significantPoints.get(i)).getNormal().x,
                                facet.getVertex(significantPoints.get(i)).getNormal().y,
                                facet.getVertex(significantPoints.get(i)).getNormal().z),null, null);
                        } else {
                            ni = new MeshPoint (new Vector3d(normals[significantPoints.get(i)].getPosition().x,normals[significantPoints.get(i)].getPosition().y,
                                normals[significantPoints.get(i)].getPosition().z), null, null);
                        }
                        ni = ni.divide(ni.abs());
                        
                        MeshPoint nj;
                        if(facet.getVertex(significantPoints.get(j)).getNormal() != null) {
                            nj = new MeshPoint(new Vector3d(facet.getVertex(significantPoints.get(j)).getNormal().x,
                                facet.getVertex(significantPoints.get(j)).getNormal().y,
                                facet.getVertex(significantPoints.get(j)).getNormal().z),null, null);
                        } else {
                            nj = new MeshPoint (new Vector3d(normals[significantPoints.get(j)].getPosition().x,normals[significantPoints.get(j)].getPosition().y,
                            normals[significantPoints.get(j)].getPosition().z), null, null);
                        }
                        
                        nj = nj.divide(nj.abs());*/
                        
                        ni = new MeshPoint (new Vector3d(normals[significantPoints.get(i)].getPosition().x,normals[significantPoints.get(i)].getPosition().y,
                                normals[significantPoints.get(i)].getPosition().z), null, null);
                        
                        ni = ni.dividePosition(ni.abs());
                        
                        MeshPoint nj;
                        nj = new MeshPoint (new Vector3d(normals[significantPoints.get(j)].getPosition().x,normals[significantPoints.get(j)].getPosition().y,
                            normals[significantPoints.get(j)].getPosition().z), null, null);
                        
                        nj = nj.dividePosition(nj.abs());
                        
                        MeshPoint normVec = ni.subtractPosition(nj);
                        normVec = normVec.dividePosition(normVec.abs());
                        double normCos = normVec.dotProduct(normal);

                        if (Math.abs(normCos) >= conf.getMinNormAngleCos()) {
                            
                            Plane newPlane = new Plane(normal.getPosition().x, normal.getPosition().y, normal.getPosition().z, d);
                            int currentVotes = GetVotes(
                                newPlane, 
                                significantCurvatures, 
                                significantPoints, 
                                0.5, 
                                conf.getMinAngleCos(), 
                                conf.getMinNormAngleCos(), 
                                boundingBox.getMaxDiag() * conf.getMaxRelDistance());
                            

                            planes.add(new AproxSymmetryPlane(newPlane, currentVotes));
                            
                            if (currentVotes > lastVotes) {
                                lastVotes = currentVotes;
                                plane = newPlane;
                            }
                        }
                        
                    }
                }
            }
            progressMonitor.setNote("Task step: " + (int) ((i + 1) * percentsPerStep));
            progressMonitor.setProgress(i);
        };
        
        Collections.sort(planes);
        ArrayList<AproxSymmetryPlane> finalPlanes = new ArrayList<>();
        for (int i = 0; i < planes.size(); i++) {
            if (planes.get(i).votes == lastVotes){
                finalPlanes.add(planes.get(i));
            }
        }
        Plane finalPlane = new Plane(0, 0, 0, 0);
        //v plane nic neni 
        MeshPoint refDir = new MeshPoint(new Vector3d(finalPlanes.get(0).a, finalPlanes.get(0).b, finalPlanes.get(0).c), null, null);
        for (int i = 0; i < finalPlanes.size(); i++) {
            MeshPoint normDir = new MeshPoint(new Vector3d(finalPlanes.get(i).a, finalPlanes.get(i).b, finalPlanes.get(i).c), null, null);
            if (normDir.dotProduct(refDir) < 0) {
                finalPlane.a -= finalPlanes.get(i).a;
                finalPlane.b -= finalPlanes.get(i).b;
                finalPlane.c -= finalPlanes.get(i).c;
                finalPlane.d -= finalPlanes.get(i).d;
            }
            else {
                finalPlane.a += finalPlanes.get(i).a;
                finalPlane.b += finalPlanes.get(i).b;
                finalPlane.c += finalPlanes.get(i).c;
                finalPlane.d += finalPlanes.get(i).d;
            }
        }
        finalPlane.normalize();
        if (conf.isAveraging()){
            plane = finalPlane;
        }
        JOptionPane.showMessageDialog(frameMain, "Final plane: " + plane.a + " " + plane.b + " " + plane.c + " " +
                plane.d + "\n" + "Votes: " + lastVotes, "Symmetry estimate done.", 0,
                new ImageIcon(getClass().getResource("/cz/fidentis/analyst/gui/resources/exportedModel.png")));

        progressMonitor.close();
        return plane;
    }
    
    /**
     * 
     * @param plane Plane computed as symmetry plane
     * @return mesh that represents facet with computed plane of approximate symmetry
     */
    public SymmetryCounter mergeWithPlane(Plane plane) {
        MeshPoint normal = new MeshPoint(new Vector3d(plane.a, plane.b, plane.c), null, null);
        MeshPoint midPoint = boundingBox.getMidPoint();

        double alpha = -((plane.a * midPoint.getPosition().x) + 
                (plane.b * midPoint.getPosition().y) + (plane.c * midPoint.getPosition().z) +
                plane.d) / (normal.dotProduct(normal));

        MeshPoint midPointOnPlane = midPoint.addPosition(normal.multiplyPosition(alpha));

        double val = plane.a * midPointOnPlane.getPosition().x + plane.b *
                midPointOnPlane.getPosition().y + plane.c *
                midPointOnPlane.getPosition().z + plane.d;

        MeshPoint a;
        if (Math.abs(normal.dotProduct(new MeshPoint(new Vector3d(0.0, 1.0, 0.0), null, null))) 
                > Math.abs(normal.dotProduct(new MeshPoint(new Vector3d (1.0, 0.0, 0.0), null, null)))) {
                a = normal.crossProduct(new MeshPoint(new Vector3d(1.0, 0.0, 0.0), null, null));
        }
        else {
            a = normal.crossProduct(new MeshPoint(new Vector3d(0.0, 1.0, 0.0), null, null));
        }
        a = a.dividePosition(a.abs());

        MeshPoint b = normal.crossProduct(a);
        b = b.dividePosition(b.abs());
       

        SymmetryCounter planeMesh = Plane.createPlaneMesh(midPointOnPlane, a, b,
                (boundingBox.getMaxPoint().subtractPosition(boundingBox.getMinPoint())).getPosition().x);


        return mergeMeshWith(planeMesh);
    }
    
    /**
     * 
     * @param s mesh that will be merged
     * @return mesh with merged vertices from both meshes 
     */
    public SymmetryCounter mergeMeshWith(SymmetryCounter s) {
        CornerTableRow row1 = new CornerTableRow(facet.getNumberOfVertices(), -1);
        CornerTableRow row2 = new CornerTableRow(facet.getNumberOfVertices() + 1, facet.getNumberOfVertices() + 3);
        CornerTableRow row3 = new CornerTableRow(facet.getNumberOfVertices() + 2, -1);
        CornerTableRow row4 = new CornerTableRow(facet.getNumberOfVertices() + 2, -1);
        CornerTableRow row5 = new CornerTableRow(facet.getNumberOfVertices() + 3, facet.getNumberOfVertices() + 1);
        CornerTableRow row6 = new CornerTableRow(facet.getNumberOfVertices(), -1);
        
        facet.getCornerTable().addRow(row1);
        facet.getCornerTable().addRow(row2);
        facet.getCornerTable().addRow(row3);
        facet.getCornerTable().addRow(row4);
        facet.getCornerTable().addRow(row5);
        facet.getCornerTable().addRow(row6);

        for(int n = 0; n < 4; n++) {
            facet.addVertex(s.getFacet().getVertices().get(n));
        }        
        return this;
    }
}
