package cz.fidentis.analyst.symmetry;

import cz.fidentis.analyst.mesh.core.MeshPoint;
import java.util.List;
import javax.vecmath.Vector3d;

/**
 * @author Natália Bebjaková
 * 
 * Represent min-max box.
 * It is automatically maintained by given point array.
 * 
 */

public class BoundingBox {
    private boolean isMinMaxValid;
    private MeshPoint maxPoint;
    private MeshPoint minPoint;
    private MeshPoint midPoint;
    private double maxDiag;
    private final List<MeshPoint> points;
    
    /**
     * 
     * @return max point of the bounding box
     */
    public MeshPoint getMaxPoint() {
        return maxPoint;
    }
    
    /**
     * 
     * @param point new max point of the bounding box
     */
    public void setMaxPoint(MeshPoint point) {
        this.maxPoint = point;
    }
    
    /**
     * 
     * @return middle point of the bounding box
     */
    public MeshPoint getMidPoint() {
        return midPoint;
    }
    
    /**
     * 
     * @param point new middle point of the bounding box
     */
    public void setMidPoint(MeshPoint point) {
        this.midPoint = point;
    }
    
    /**
     * 
     * @return min point of the bounding box
     */
    public MeshPoint getMinPoint() {
        return minPoint;
    }
    
    /**
     * 
     * @param point new min point of the bounding box
     */
    public void setMinPoint(MeshPoint point) {
        this.minPoint = point;
    }
    
    /** 
     * Creates bounding box that is automatically maintained with respect to given array.
     * @param points array of points
     */
    public BoundingBox(List<MeshPoint> points) {
        this.points = points;
        this.ValidateMinMax();
        this.ValidateMidDiag();
    }        

    /**
     * @return point array that is managed by curent bounding box.
    */
    public List<MeshPoint> getPoints() {
        return points;
    }
    
    /**
     * Recomputes the BoundingBox from all points
    */
    private void ValidateMinMax() {
        minPoint = new MeshPoint(new Vector3d(Double.MAX_VALUE,Double.MAX_VALUE,Double.MAX_VALUE), null, null);
        maxPoint = new MeshPoint(new Vector3d(Double.MIN_VALUE,Double.MIN_VALUE,Double.MIN_VALUE), null, null);
 
        for (int i = 0; i < points.size(); i++) {
            MeshPoint point = points.get(i);
            
            minPoint.getPosition().x = Math.min(minPoint.getPosition().x, point.getPosition().x);
            minPoint.getPosition().y = Math.min(minPoint.getPosition().y, point.getPosition().y);
            minPoint.getPosition().z = Math.min(minPoint.getPosition().z, point.getPosition().z);
            
            maxPoint.getPosition().x = Math.max(maxPoint.getPosition().x, point.getPosition().x);
            maxPoint.getPosition().y = Math.max(maxPoint.getPosition().y, point.getPosition().y);
            maxPoint.getPosition().z = Math.max(maxPoint.getPosition().z, point.getPosition().z);                    
	}
	isMinMaxValid = true;
	this.ValidateMidDiag();
    }

    /**
     * Recompute mid-point and max diagonal length.
     */   
    private void ValidateMidDiag() {
        midPoint = (minPoint.addPosition(maxPoint)).multiplyPosition(0.5);
        MeshPoint diag = maxPoint.subtractPosition(minPoint);
        this.maxDiag = diag.abs();
    }

    /**
     * Return volume diagonal of the bounding box.
     * @return maximal diagonal of bounding box
     */
    public double getMaxDiag() {
        if (!isMinMaxValid) {
            this.ValidateMinMax();
        }
        return maxDiag;
    }
    
    /**
     * Returns description of BoundignBox.
     * @return String representation of the bounding box
     */
    @Override
    public String toString() {
        String str = "BoundingBox: ";
        if (this.minPoint == null || this.maxPoint == null || this.midPoint == null) {
            return str += "undefined (there are no points)";
        }
        str += "\n";
        str += "\t" + "- min point : " + this.minPoint + "\n";
        str += "\t" + "- max point : " + this.maxPoint + "\n";
        str += "\t" + "- mid point : " + this.midPoint + "\n";
        str += "\t" + "- max diag  : " + this.maxDiag + "\n";
        return str;
    }      
}  