package cz.fidentis.analyst.gui;

import static cz.fidentis.analyst.gui.UserInterface.frameMain;
import cz.fidentis.analyst.mesh.core.MeshModel;
import cz.fidentis.analyst.symmetry.Config;
import cz.fidentis.analyst.symmetry.Plane;
import cz.fidentis.analyst.symmetry.SymmetryCounter;
import java.awt.Cursor;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JSlider;
import javax.swing.JTextField;
import javax.swing.event.ChangeEvent;

/**
 *
 * @author Natália Bebjaková
 * 
 * Panel for estimating approximate symmetry of the model
 */
public final class SymmetryPanel extends javax.swing.JPanel {
    /**
     * Configuration with default values 
     */
    private Config config;
    /**
     * GL Canvas on which model is displayed
     */
    private Canvas canvas;
    /**
     * Class that is responsible for computing the symmetry
     */
    private SymmetryCounter symCounter;
    /**
     * Computed approximate plane of the symmetry
     */
    private Plane finalPlane;

    /**
     * 
     * @return GL Canvas
     */
    public Canvas getCanvas() {
        return canvas;
    }

    /**
     * 
     * @param canvas GL Canvas
     */
    public void setCanvas(Canvas canvas) {
        this.canvas = canvas;
    }

    /**
     * 
     * @return Configuration for computing symmetry
     */
    public Config getConfig() {
        return config;
    }

    /**
     * 
     * @param config Configuration for computing symmetry
     */
    public void setConfig(Config config) {
        this.config = config;
    }
    
    /**
     * 
     * @return Button for returning original model from loaded file 
     */
    public JLabel getOriginalModelButton() {
        return originalModelButton;
    }
    
    /**
     * 
     * @return Label for showing final plane of symmetry
     */
    public JLabel getShowPlaneLabel() {
        return showPlaneLabel;
    }
        
    /**
     * Sets configuration values according to text fields on panel
     * User can change this text fields
     */
    public void setConfigParams() {
        config.setMaxRelDistance(Double.parseDouble(distanceTextField.getText()));
        config.setMinAngleCos(Double.parseDouble(textFieldMinCos.getText()));
        config.setMinCurvRatio(Double.parseDouble(textFieldCurvature.getText()));
        config.setMinNormAngleCos(Double.parseDouble(normalTextField.getText()));
        config.setSignificantPointCount(Integer.parseInt(significantTextField.getText()));
    }
    
    /**
     * Sets values in text field according to configuration
     */
    public void setTextFieldsDueToConfig() {
        distanceTextField.setText(Double.toString(config.getMaxRelDistance()));
        textFieldMinCos.setText(Double.toString(config.getMinAngleCos()));
        textFieldCurvature.setText(Double.toString(config.getMinCurvRatio()));
        normalTextField.setText(Double.toString(config.getMinNormAngleCos()));
        significantTextField.setText(Integer.toString(config.getSignificantPointCount()));    
    }
    
    /**
     * 
     * @param slider Slider 
     * @param field text field which belongs to slider
     */
    public void setSlider(JSlider slider, JTextField field) {
        slider.setValue((int) (Double.parseDouble(field.getText()) * 100));
        
        slider.addChangeListener((ChangeEvent ce) -> {
            field.setText(""+slider.getValue()/100.0);
            defaultValues.setVisible(true);
        });
        
    }
    
    /**
     * Creates new form symmetryPanel
     */
    public SymmetryPanel() {
        initComponents();
        config = Config.getDefault();
        
        setSlider(relativeDistanceSlider, distanceTextField);
        setSlider(curavatureSlider, textFieldCurvature);
        setSlider(angleCosineSlider, textFieldMinCos);
        setSlider(normalAngleSlider, normalTextField);
                
        significantPointSlider.setValue((int) (Double.parseDouble(significantTextField.getText())));
        significantPointSlider.addChangeListener((ChangeEvent ce) -> {
            significantTextField.setText("" + significantPointSlider.getValue());
        });
        originalModelButton.setVisible(false);
        showPlaneLabel.setVisible(false);
    }


    /**
     * Calculate approxy symmetry of the model 
     * Accuracy of the symmetry plane is influenced by configuration 
     * 
     * @throws InterruptedException exception can be thrown beacause of progress monitor
     */
    private void countSymmetry() throws InterruptedException {
        MeshModel model = new MeshModel();
        symCounter = new SymmetryCounter(canvas.getModel().getFacets().get(0));
        symCounter.init();
        symCounter.setPanel(this);
        finalPlane = symCounter.getAproxSymmetryPlane(config);
        SymmetryCounter counted = symCounter.mergeWithPlane(finalPlane);
        model.addFacet(counted.getFacet());
        this.canvas.changeModel(model);  
    }


    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     * 
     * Code generated by NetBeans
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        symetrySpecificationPanel = new javax.swing.JPanel();
        curavatureSlider = new javax.swing.JSlider();
        angleCosineSlider = new javax.swing.JSlider();
        minCurvatio = new javax.swing.JLabel();
        minCurvatio2 = new javax.swing.JLabel();
        normalAngleSlider = new javax.swing.JSlider();
        minCurvatio3 = new javax.swing.JLabel();
        significantPointSlider = new javax.swing.JSlider();
        minCurvatio4 = new javax.swing.JLabel();
        relativeDistanceSlider = new javax.swing.JSlider();
        significantPointLabel = new javax.swing.JLabel();
        symetryButton = new javax.swing.JLabel();
        averagingCheckBox = new javax.swing.JCheckBox();
        textFieldCurvature = new javax.swing.JTextField();
        textFieldMinCos = new javax.swing.JTextField();
        normalTextField = new javax.swing.JTextField();
        distanceTextField = new javax.swing.JTextField();
        significantTextField = new javax.swing.JTextField();
        minCurvatio8 = new javax.swing.JLabel();
        jLabel1 = new javax.swing.JLabel();
        originalModelButton = new javax.swing.JLabel();
        showPlaneLabel = new javax.swing.JLabel();
        defaultValues = new javax.swing.JLabel();

        symetrySpecificationPanel.setBackground(new java.awt.Color(176, 230, 226));

        curavatureSlider.setBackground(new java.awt.Color(250, 250, 250));
        curavatureSlider.setMajorTickSpacing(1);
        curavatureSlider.setMinimum(50);
        curavatureSlider.setSnapToTicks(true);
        curavatureSlider.setCursor(new java.awt.Cursor(java.awt.Cursor.HAND_CURSOR));
        curavatureSlider.setOpaque(false);

        angleCosineSlider.setMinimum(80);
        angleCosineSlider.setSnapToTicks(true);
        angleCosineSlider.setCursor(new java.awt.Cursor(java.awt.Cursor.HAND_CURSOR));
        angleCosineSlider.setOpaque(false);

        minCurvatio.setFont(new java.awt.Font("Neue Haas Unica Pro", 0, 14)); // NOI18N
        minCurvatio.setForeground(new java.awt.Color(20, 114, 105));
        minCurvatio.setText("Min. Curvature Ratio");

        minCurvatio2.setFont(new java.awt.Font("Neue Haas Unica Pro", 0, 14)); // NOI18N
        minCurvatio2.setForeground(new java.awt.Color(20, 114, 105));
        minCurvatio2.setText("Min. Angle Cosine");

        normalAngleSlider.setMinimum(80);
        normalAngleSlider.setSnapToTicks(true);
        normalAngleSlider.setCursor(new java.awt.Cursor(java.awt.Cursor.HAND_CURSOR));
        normalAngleSlider.setOpaque(false);

        minCurvatio3.setFont(new java.awt.Font("Neue Haas Unica Pro", 0, 14)); // NOI18N
        minCurvatio3.setForeground(new java.awt.Color(20, 114, 105));
        minCurvatio3.setText("Max. Relative Distance");

        significantPointSlider.setMajorTickSpacing(100);
        significantPointSlider.setMaximum(300);
        significantPointSlider.setSnapToTicks(true);
        significantPointSlider.setCursor(new java.awt.Cursor(java.awt.Cursor.HAND_CURSOR));
        significantPointSlider.setOpaque(false);

        minCurvatio4.setFont(new java.awt.Font("Neue Haas Unica Pro", 0, 14)); // NOI18N
        minCurvatio4.setForeground(new java.awt.Color(20, 114, 105));
        minCurvatio4.setText("Min. Normal Angle Cosine");

        relativeDistanceSlider.setMaximum(5);
        relativeDistanceSlider.setSnapToTicks(true);
        relativeDistanceSlider.setCursor(new java.awt.Cursor(java.awt.Cursor.HAND_CURSOR));
        relativeDistanceSlider.setOpaque(false);

        significantPointLabel.setFont(new java.awt.Font("Neue Haas Unica Pro", 0, 14)); // NOI18N
        significantPointLabel.setForeground(new java.awt.Color(20, 114, 105));
        significantPointLabel.setText("Significant Point Count");

        symetryButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/cz/fidentis/analyst/gui/resources/symetryCount.png"))); // NOI18N
        symetryButton.addMouseMotionListener(new java.awt.event.MouseMotionAdapter() {
            public void mouseMoved(java.awt.event.MouseEvent evt) {
                symetryButtonMouseMoved(evt);
            }
        });
        symetryButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                symetryButtonMouseClicked(evt);
            }
            public void mouseExited(java.awt.event.MouseEvent evt) {
                symetryButtonMouseExited(evt);
            }
        });

        averagingCheckBox.setSelected(true);
        averagingCheckBox.setOpaque(false);
        averagingCheckBox.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                averagingCheckBoxMouseClicked(evt);
            }
        });

        textFieldCurvature.setText("0.8");
        textFieldCurvature.setToolTipText("");

        textFieldMinCos.setText("0.985");
        textFieldMinCos.setToolTipText("");

        normalTextField.setText("0.985");
        normalTextField.setToolTipText("");

        distanceTextField.setText("0.01");
        distanceTextField.setToolTipText("");

        significantTextField.setText("200");
        significantTextField.setToolTipText("");

        minCurvatio8.setFont(new java.awt.Font("Neue Haas Unica Pro", 0, 14)); // NOI18N
        minCurvatio8.setForeground(new java.awt.Color(20, 114, 105));
        minCurvatio8.setText("Averaging");

        originalModelButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/cz/fidentis/analyst/gui/resources/originalModel.png"))); // NOI18N
        originalModelButton.setCursor(new java.awt.Cursor(java.awt.Cursor.HAND_CURSOR));
        originalModelButton.addMouseMotionListener(new java.awt.event.MouseMotionAdapter() {
            public void mouseMoved(java.awt.event.MouseEvent evt) {
                originalModelButtonMouseMoved(evt);
            }
        });
        originalModelButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                originalModelButtonMouseClicked(evt);
            }
            public void mouseExited(java.awt.event.MouseEvent evt) {
                originalModelButtonMouseExited(evt);
            }
        });

        showPlaneLabel.setIcon(new javax.swing.ImageIcon(getClass().getResource("/cz/fidentis/analyst/gui/resources/showPlane.png"))); // NOI18N
        showPlaneLabel.setCursor(new java.awt.Cursor(java.awt.Cursor.HAND_CURSOR));
        showPlaneLabel.setDoubleBuffered(true);
        showPlaneLabel.addMouseMotionListener(new java.awt.event.MouseMotionAdapter() {
            public void mouseMoved(java.awt.event.MouseEvent evt) {
                showPlaneLabelMouseMoved(evt);
            }
        });
        showPlaneLabel.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                showPlaneLabelMouseClicked(evt);
            }
            public void mouseExited(java.awt.event.MouseEvent evt) {
                showPlaneLabelMouseExited(evt);
            }
        });

        defaultValues.setFont(new java.awt.Font("Neue Haas Unica Pro", 0, 14)); // NOI18N
        defaultValues.setForeground(new java.awt.Color(20, 114, 105));
        defaultValues.setText("Default values");
        defaultValues.setCursor(new java.awt.Cursor(java.awt.Cursor.HAND_CURSOR));
        defaultValues.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                defaultValuesMouseClicked(evt);
            }
        });

        javax.swing.GroupLayout symetrySpecificationPanelLayout = new javax.swing.GroupLayout(symetrySpecificationPanel);
        symetrySpecificationPanel.setLayout(symetrySpecificationPanelLayout);
        symetrySpecificationPanelLayout.setHorizontalGroup(
            symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(symetrySpecificationPanelLayout.createSequentialGroup()
                .addGap(17, 17, 17)
                .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(symetrySpecificationPanelLayout.createSequentialGroup()
                        .addGap(0, 0, Short.MAX_VALUE)
                        .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                            .addGroup(symetrySpecificationPanelLayout.createSequentialGroup()
                                .addComponent(normalAngleSlider, javax.swing.GroupLayout.PREFERRED_SIZE, 200, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                                .addComponent(normalTextField, javax.swing.GroupLayout.PREFERRED_SIZE, 46, javax.swing.GroupLayout.PREFERRED_SIZE))
                            .addGroup(symetrySpecificationPanelLayout.createSequentialGroup()
                                .addComponent(significantPointSlider, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                                .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                                    .addComponent(textFieldMinCos, javax.swing.GroupLayout.DEFAULT_SIZE, 46, Short.MAX_VALUE)
                                    .addComponent(textFieldCurvature)
                                    .addComponent(distanceTextField)
                                    .addComponent(significantTextField))))
                        .addGap(63, 63, 63))
                    .addGroup(symetrySpecificationPanelLayout.createSequentialGroup()
                        .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(minCurvatio, javax.swing.GroupLayout.Alignment.TRAILING)
                            .addComponent(minCurvatio2, javax.swing.GroupLayout.Alignment.TRAILING)
                            .addComponent(minCurvatio4, javax.swing.GroupLayout.Alignment.TRAILING)
                            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                .addComponent(significantPointLabel)
                                .addComponent(minCurvatio3))
                            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, symetrySpecificationPanelLayout.createSequentialGroup()
                                .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                                    .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                        .addComponent(jLabel1)
                                        .addGroup(symetrySpecificationPanelLayout.createSequentialGroup()
                                            .addGap(80, 80, 80)
                                            .addComponent(minCurvatio8)
                                            .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                                            .addComponent(averagingCheckBox)))
                                    .addComponent(showPlaneLabel))
                                .addGap(9, 9, 9)))
                        .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, symetrySpecificationPanelLayout.createSequentialGroup()
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                                .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                    .addComponent(symetryButton, javax.swing.GroupLayout.Alignment.TRAILING)
                                    .addComponent(originalModelButton, javax.swing.GroupLayout.Alignment.TRAILING, javax.swing.GroupLayout.PREFERRED_SIZE, 181, javax.swing.GroupLayout.PREFERRED_SIZE)
                                    .addComponent(defaultValues, javax.swing.GroupLayout.Alignment.TRAILING))
                                .addGap(63, 63, 63))
                            .addGroup(symetrySpecificationPanelLayout.createSequentialGroup()
                                .addGap(56, 56, 56)
                                .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                                    .addComponent(angleCosineSlider, javax.swing.GroupLayout.PREFERRED_SIZE, 200, javax.swing.GroupLayout.PREFERRED_SIZE)
                                    .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                        .addComponent(relativeDistanceSlider, javax.swing.GroupLayout.PREFERRED_SIZE, 200, javax.swing.GroupLayout.PREFERRED_SIZE)
                                        .addComponent(curavatureSlider, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)))
                                .addGap(0, 122, Short.MAX_VALUE))))))
        );
        symetrySpecificationPanelLayout.setVerticalGroup(
            symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(symetrySpecificationPanelLayout.createSequentialGroup()
                .addGap(72, 72, 72)
                .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                    .addGroup(symetrySpecificationPanelLayout.createSequentialGroup()
                        .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                .addComponent(minCurvatio, javax.swing.GroupLayout.PREFERRED_SIZE, 26, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addComponent(textFieldCurvature, javax.swing.GroupLayout.Alignment.TRAILING, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                            .addGroup(symetrySpecificationPanelLayout.createSequentialGroup()
                                .addGap(4, 4, 4)
                                .addComponent(curavatureSlider, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)))
                        .addGap(36, 36, 36)
                        .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(angleCosineSlider, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(minCurvatio2, javax.swing.GroupLayout.PREFERRED_SIZE, 26, javax.swing.GroupLayout.PREFERRED_SIZE)))
                    .addComponent(textFieldMinCos, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(symetrySpecificationPanelLayout.createSequentialGroup()
                        .addGap(18, 18, 18)
                        .addComponent(minCurvatio4, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                        .addGap(22, 22, 22))
                    .addGroup(symetrySpecificationPanelLayout.createSequentialGroup()
                        .addGap(34, 34, 34)
                        .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(normalTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(normalAngleSlider, javax.swing.GroupLayout.PREFERRED_SIZE, 30, javax.swing.GroupLayout.PREFERRED_SIZE))
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, 34, Short.MAX_VALUE)))
                .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                    .addComponent(minCurvatio3, javax.swing.GroupLayout.PREFERRED_SIZE, 26, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                        .addComponent(relativeDistanceSlider, javax.swing.GroupLayout.Alignment.TRAILING, javax.swing.GroupLayout.PREFERRED_SIZE, 0, Short.MAX_VALUE)
                        .addComponent(distanceTextField, javax.swing.GroupLayout.Alignment.TRAILING)))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, 31, Short.MAX_VALUE)
                .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(significantTextField, javax.swing.GroupLayout.Alignment.TRAILING, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(significantPointLabel, javax.swing.GroupLayout.Alignment.TRAILING, javax.swing.GroupLayout.PREFERRED_SIZE, 14, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(significantPointSlider, javax.swing.GroupLayout.Alignment.TRAILING, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(14, 14, 14)
                .addComponent(defaultValues)
                .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                    .addGroup(symetrySpecificationPanelLayout.createSequentialGroup()
                        .addGroup(symetrySpecificationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                            .addComponent(averagingCheckBox)
                            .addComponent(minCurvatio8, javax.swing.GroupLayout.PREFERRED_SIZE, 28, javax.swing.GroupLayout.PREFERRED_SIZE))
                        .addGap(65, 65, 65)
                        .addComponent(jLabel1)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                        .addComponent(showPlaneLabel))
                    .addGroup(symetrySpecificationPanelLayout.createSequentialGroup()
                        .addGap(3, 3, 3)
                        .addComponent(symetryButton, javax.swing.GroupLayout.PREFERRED_SIZE, 75, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(originalModelButton)))
                .addContainerGap(32, Short.MAX_VALUE))
        );

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 580, Short.MAX_VALUE)
            .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                .addGroup(layout.createSequentialGroup()
                    .addComponent(symetrySpecificationPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addGap(0, 0, Short.MAX_VALUE)))
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 554, Short.MAX_VALUE)
            .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                .addComponent(symetrySpecificationPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );
    }// </editor-fold>//GEN-END:initComponents

    /**
     * 
     * @param evt Decides if averaging is ON or OFF
     */
    private void averagingCheckBoxMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_averagingCheckBoxMouseClicked
        if(config.isAveraging()) {
            config.setAveraging(false);
        } else {
            config.setAveraging(true);
        }
    }//GEN-LAST:event_averagingCheckBoxMouseClicked

    private void symetryButtonMouseExited(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_symetryButtonMouseExited
        symetryButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/cz/fidentis/analyst/gui/resources/symetryCount.png")));
    }//GEN-LAST:event_symetryButtonMouseExited

    /**
     * 
     * @param evt Symmetry is estimated. If model is not loaded, user is warned 
     */
    private void symetryButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_symetryButtonMouseClicked
        setConfigParams();
        if (canvas.getModel().getFacets().isEmpty()){
            JOptionPane.showMessageDialog(frameMain, "You have to load the model.", "Model not loaded",
                    0, new ImageIcon(getClass().getResource("/cz/fidentis/analyst/gui/resources/notLoadedModel.png")));
        } else {
            try {
                countSymmetry();
            } catch (InterruptedException ex) {
                Logger.getLogger(SymmetryPanel.class.getName()).log(Level.SEVERE, null, ex);
            }
            originalModelButton.setVisible(true);
            showPlaneLabel.setVisible(true);
        }
    }//GEN-LAST:event_symetryButtonMouseClicked

    /**
    * 
    * @param evt Changes button
    */
    private void symetryButtonMouseMoved(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_symetryButtonMouseMoved
        symetryButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/cz/fidentis/analyst/gui/resources/symetryCountClicked.png")));
        symetryButton.setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
    }//GEN-LAST:event_symetryButtonMouseMoved

    /**
     * 
     * @param Changes button
     */
    private void originalModelButtonMouseMoved(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_originalModelButtonMouseMoved
        originalModelButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/cz/fidentis/analyst/gui/resources/originalModelPressed.png")));
    }//GEN-LAST:event_originalModelButtonMouseMoved

    /**
     * 
     * @param evt Changes button
     */
    private void originalModelButtonMouseExited(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_originalModelButtonMouseExited
        originalModelButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/cz/fidentis/analyst/gui/resources/originalModel.png")));
    }//GEN-LAST:event_originalModelButtonMouseExited

    /**
     * 
     * @param evt Original model (without plane) is displayed
     */
    private void originalModelButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_originalModelButtonMouseClicked
        canvas.changeModel(canvas.getLoadedModel());
        originalModelButton.setVisible(false);
        showPlaneLabel.setVisible(false);
    }//GEN-LAST:event_originalModelButtonMouseClicked

    /**
     * 
     * @param evt Changes button
     */
    private void showPlaneLabelMouseMoved(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_showPlaneLabelMouseMoved
        showPlaneLabel.setIcon(new javax.swing.ImageIcon(getClass().getResource("/cz/fidentis/analyst/gui/resources/showPlanePressed.png")));
    }//GEN-LAST:event_showPlaneLabelMouseMoved

    /**
     * 
     * @param evt Final computed plane is shown to user
     */
    private void showPlaneLabelMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_showPlaneLabelMouseClicked
        JOptionPane.showMessageDialog(frameMain, "Approximate plane of symmetry:  \n" + finalPlane.a + "\n" + finalPlane.b + "\n" + finalPlane.c + "\n" +
            finalPlane.d + "\n", "Final plane.", 0, new ImageIcon(getClass().getResource("/cz/fidentis/analyst/gui/resources/showPlanePane.png")));
    }//GEN-LAST:event_showPlaneLabelMouseClicked

    /**
     * 
     * @param evt Changes button
     */
    private void showPlaneLabelMouseExited(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_showPlaneLabelMouseExited
        showPlaneLabel.setIcon(new javax.swing.ImageIcon(getClass().getResource("/cz/fidentis/analyst/gui/resources/showPlane.png")));
    }//GEN-LAST:event_showPlaneLabelMouseExited

    /**
     * 
     * @param evt values of configuration back to deafult 
     */
    private void defaultValuesMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_defaultValuesMouseClicked
        config = Config.getDefault();
        setTextFieldsDueToConfig();
    }//GEN-LAST:event_defaultValuesMouseClicked
    


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JSlider angleCosineSlider;
    private javax.swing.JCheckBox averagingCheckBox;
    private javax.swing.JSlider curavatureSlider;
    private javax.swing.JLabel defaultValues;
    private javax.swing.JTextField distanceTextField;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel minCurvatio;
    private javax.swing.JLabel minCurvatio2;
    private javax.swing.JLabel minCurvatio3;
    private javax.swing.JLabel minCurvatio4;
    private javax.swing.JLabel minCurvatio8;
    private javax.swing.JSlider normalAngleSlider;
    private javax.swing.JTextField normalTextField;
    private javax.swing.JLabel originalModelButton;
    private javax.swing.JSlider relativeDistanceSlider;
    private javax.swing.JLabel showPlaneLabel;
    private javax.swing.JLabel significantPointLabel;
    private javax.swing.JSlider significantPointSlider;
    private javax.swing.JTextField significantTextField;
    private javax.swing.JLabel symetryButton;
    private javax.swing.JPanel symetrySpecificationPanel;
    private javax.swing.JTextField textFieldCurvature;
    private javax.swing.JTextField textFieldMinCos;
    // End of variables declaration//GEN-END:variables
}