package cz.fidentis.analyst.mesh;

import cz.fidentis.analyst.mesh.core.MeshFacet;
import java.util.concurrent.Callable;

/**
 * A functor. When instantiated, it can be gradually applied to multiple mesh facets.
 * It inspects the state of the mesh facets one by one, and (cumulatevely) computes results.
 * <p>
 * Implement this interface whenever you want to define new algorithm over a mesh.
 * </p>
 * <p>
 * The visitor can be instatiated either as synchronous or asynchronous. 
 * The {@link cz.fidentis.analyst.mesh.MeshVisitor#visit} inspection method
 * of a synchronous visitor is applied immediately. On the contrary, if the
 * visitor is asynchronous, then the method only stores the inpected
 * mesh for later concurrent inspection. To trigger the inspection,
 * a {@link java.util.concurrent.ExecutorService} has to be used calling
 * the {@link cz.fidentis.analyst.mesh.MeshVisitor#call} method asynchronously
 * and then waiting for the results of all triggered visitors. 
 * </p>
 * <p>
 * If the visitor is thread-safe, then a single instance of the visitor
 * can visit concurrenly (and assynchronously) multiple mesh facets. Otherwise, 
 * the parallel inspection is still possible, but a new instance of the visitor 
 * has to be used for each mesh facet.
 * </p>
 * <p>
 * The {@link cz.fidentis.analyst.mesh.core.MeshModel#compute} method takes 
 * these properties into account. If the visitor is asynchronous and thread-safe, 
 * then it is applied concurrently to all mesh facets of the mesh model.
 * </p>
 * 
 * @author Radek Oslejsek
 */
public abstract class MeshVisitor implements Callable<MeshVisitor> {
    
    private boolean asynchronous;
    
    /**
     * Facet stored temporarily for later cuncurrent inspection via 
     * the {@link cz.fidentis.analyst.mesh.MeshVisitor#call} method.
     */
    private MeshFacet facet;
    
    
    /**
     * Constructor.
     * 
     * @param asynchronous If {@code true}, then asynchronous visitor is created.
     * (the {@link cz.fidentis.analyst.mesh.MeshVisitor#visit} is not executed 
     * immediately but postponed for concurrent asynchronous execution). 
     * Otherwise, the visitor is created as sequential 
     * (the {@link cz.fidentis.analyst.mesh.MeshVisitor#visit} is executed immediately.)
     */
    public MeshVisitor(boolean asynchronous) {
        this.asynchronous = asynchronous;
    }
    
    /**
     * Returns {@code true} if the implementation is thread-safe and then
     * <b>a single visitor instance</b> can be applied to multiple mesh facets simultaneously.
     * If the visitor is not thread-safe, the concurrent inspection 
     * via the {@link cz.fidentis.analyst.mesh.MeshVisitor#visit}
     * is still possible, but with multiple visitor instances - one for each facet.
     * <p>
     * Thread-safe implementation means that any read or write from/to the visitor's 
     * state is implemented as {@code synchronized}.
     * </p>
     * 
     * @return {@code true} if the implementation is thread-safe.
     */
    public boolean isThreadSafe() {
        return true;
    }
    
    /**
     * Returns {@code true} if this visitor was created as asynchronous.
     * 
     * @return {@code true} if this visitor was created as asynchronous.
     */
    public boolean isAsynchronous() {
        return this.asynchronous;
    }
    
    /**
     * The inspection method to be implememted by specific visitors.
     * 
     * @param facet Mesh facet to be visited.
     */
    protected abstract void visitMeshFacet(MeshFacet facet);
    
    /**
     * The main visitation method, either invoked immediately (if the visitor 
     * was created a synchronous) or postponed to later parallel execution 
     * (if the visitor was created a asynchronous).
     * 
     * @param facet Mesh facet to be visited.
     */
    public void visit(MeshFacet facet) {
        if (asynchronous) {
            this.facet = facet;
        } else {
            visitMeshFacet(facet);
        }
    }
    
    @Override
    public MeshVisitor call() throws Exception {
        visitMeshFacet(facet);
        return this;
    }
}
