package cz.fidentis.analyst.face;

import com.google.common.eventbus.EventBus;
import cz.fidentis.analyst.kdtree.KdTree;
import cz.fidentis.analyst.kdtree.KdTreeEvent;
import cz.fidentis.analyst.kdtree.KdTreeListener;
import cz.fidentis.analyst.mesh.core.MeshModel;
import cz.fidentis.analyst.mesh.events.MeshEvent;
import cz.fidentis.analyst.mesh.events.MeshListener;
import cz.fidentis.analyst.mesh.io.MeshObjLoader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;

/**
 * This class encapsulates data for a 3D scan of a single human face.
 * <p>
 * Changes in the human face and its data structures (e.g., jesh model, kd-tree, etc.)
 * can be monitored by listeners. Listeners have to implement the 
 * {@link cz.fidentis.analyst.face.HumanFaceListener} iterface and they have to 
 * register using the {@link cz.fidentis.analyst.face.HumanFace#registerListener} method.
 * Then the are informed about changes in the human automatically via methods 
 * prescribed by the interface.
 * </p>
 * <p>
 * Events fired by the class:
 * <ul>
 * <li>{@link cz.fidentis.analyst.mesh.events.MeshEvent} or its sub-type if 
 * the mesh model changes (see {@link cz.fidentis.analyst.mesh.MeshModel} 
 * documentation for monitored changes).</li>
 * <li>{@link cz.fidentis.analyst.kdtree.KdTreeEvent} or its sub-type if 
 * the kd-tree changes (see {@link cz.fidentis.analyst.kdtree.KdTree} 
 * documentation for monitored changes).</li>
 * <li>{@link cz.fidentis.analyst.face.KdTreeBuiltEvent} if the kd-tree has been 
 * completely (re-)built.</li>
 * </ul>
 * </p>
 * 
 * @author Radek Oslejsek
 */
public class HumanFace implements MeshListener, KdTreeListener {
    
    private MeshModel meshModel;
    private KdTree kdTree;
    
    private EventBus eventBus = new EventBus();
    
    /**
     * Reads a 3D human phase from file.
     * 
     * @param file OBJ file
     * @throws IOException on I/O failure
     */
    public HumanFace(File file) throws IOException {
        meshModel = MeshObjLoader.read(file);
        meshModel.registerListener(this);
    }
    
    /**
     * Reads a 3D human phase from file.
     * 
     * @param is input stream with OBJ data
     * @throws IOException on I/O failure
     */
    public HumanFace(InputStream is) throws IOException {
        meshModel = MeshObjLoader.read(is);
        meshModel.registerListener(this);
    }
    
    /**
     * Returns the triangular mesh model of the human face.
     * 
     * @return the triangular mesh model of the human face
     */
    public MeshModel getMeshModel() {
        return meshModel;
    }
    
    /**
     * Builds new kd-tree and fires the {@link cz.fidentis.analyst.face.KdTreeBuiltEvent}
     * with the old tree (if any) stored in the event.
     */
    public void buildKdTree() {
        KdTree old = kdTree;
        kdTree = new KdTree(meshModel.getFacets());
        eventBus.post(new KdTreeBuiltEvent(old));
    }
    
    /**
     * Returns true if the kd-tree is computed.
     * 
     * @return true if the kd-tree is computed.
     */
    public boolean hasKdTree() {
        return kdTree != null;
    }
    
    public KdTree getKdTree() {
        return kdTree;
    }
    
    /**
     * Registers listeners (objects concerned in the human face changes) to receive events.
     * If listener is {@code null}, no exception is thrown and no action is taken.
     * 
     * @param listener Listener concerned in the human face changes.
     */
    public void registerListener(MeshListener listener) {
        eventBus.register(listener);
    }
    
    /**
     * Unregisters listeners from receiving events.
     * 
     * @param listener Registered listener
     */
    public void unregisterListener(MeshListener listener) {
        eventBus.unregister(listener);
    }

    /**
     * Captures events fired by {@link cz.fidentis.analyst.mesh.MeshModel} and
     * redirects them to our listeners.
     * 
     * @param event A fired event.
     */
    @Override
    public void meshEvent(MeshEvent event) {
        eventBus.post(event);
    }

    @Override
    public void kdTreeEvent(KdTreeEvent event) {
        eventBus.post(event);
    }
    
}
