package cz.fidentis.analyst.symmetry;

import cz.fidentis.analyst.mesh.core.CornerTableRow;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import cz.fidentis.analyst.visitors.mesh.BoundingBox;
import cz.fidentis.analyst.mesh.core.MeshFacetImpl;
import cz.fidentis.analyst.mesh.core.MeshPointImpl;
import cz.fidentis.analyst.mesh.core.MeshTriangle;
import cz.fidentis.analyst.visitors.mesh.BoundingBoxVisitor;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.swing.ImageIcon;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.ProgressMonitor;
import javax.swing.UIManager;
import javax.vecmath.Vector3d;

/**
 * Main class for computing approximate plane of symmetry of the 3D model.
 * For computing the symmetry, for every 
 * Default values of the configuration are given due to the best results on tested objects.
 * On many different 3D models, it exists other values of config that will have better impact on result. 
 * 
 * @author Natalia Bebjakova
 */
public class SymmetryEstimator {
    
    private final MeshFacet facet; // Facet of the model on which symmetry is computed 
    
    private TriangleVertexAreas[] areas; // Representation for areas of Voronoi region of triangles
    
    private BoundingBox boundingBox; // Represent min-max box. It is automatically maintained by given point array.
    
    private List<MeshTriangle> triangles; // Helping array of triangles computed from corner table 
    
    private final Config config;
    
    /**
     * Constructor.
     * 
     * @param f facet on which the symmetry will be computed 
     * @param config configuration of optional parameters of the algorithm
     */
    public SymmetryEstimator(MeshFacet f, Config config) {
        this.facet = f;
        this.config = config;
        
        this.areas = new TriangleVertexAreas[f.getNumTriangles()];
        int i = 0;
        for (MeshTriangle tri: f) {
            areas[i++] = computeTriangleVertexAreas(tri);
        }
    }
    
    /**
     * Helper constructor.
     * @param centroid Centroid
     * @param a A
     * @param b B
     * @param scale Scale
     */
    protected SymmetryEstimator(Vector3d centroid, Vector3d a, Vector3d b, double scale) {
        this(createMeshFacet(centroid, a, b, scale), Config.getDefault());
    }
    
    /**
     * 
     * @return configuration of optional parameters of the algorithm
     */
    public Config getConfig() {
        return config;
    }

    /**
     * 
     * @return Facet of the model on which symmetry is computed 
     */
    public MeshFacet getFacet() {
        return facet;
    }

    /**
     * If bounding box is not created yet, it's created now. 
     * 
     * @return Represent min-max box of the boundries of the model. 
     */
    public BoundingBox getBoundingBox() {
        if (boundingBox == null) {
            BoundingBoxVisitor visitor = new BoundingBoxVisitor(true);
            facet.accept(visitor);
            boundingBox = visitor.getBoundingBox();
        }
        return boundingBox; 
    }
    
    /**
     * Computes the approximate plane of symmetry.
     * @param panel parrent window for the progress window (can be null)
     * @return approximate plane of symmtetry
     */
    public Plane getApproxSymmetryPlane(JPanel panel) {
        
        List<ApproxSymmetryPlane> planes = new ArrayList<>();
        //List<Vector3d> normals = calculateNormals();
        
        if (!facet.hasVertexNormals()) {
            facet.calculateVertexNormals();
        }
        
        double[] curvatures = initCurvatures();
        List<Double> sortedCurvatures = new ArrayList<>();
        for (int i = 0; i < curvatures.length; i++) {
            sortedCurvatures.add(curvatures[i]);
        }
        Collections.sort(sortedCurvatures);
           
        if(config.getSignificantPointCount() > facet.getNumberOfVertices()) {
            config.setSignificantPointCount((facet.getNumberOfVertices()) - 1);
        }
        double bottomCurvature = sortedCurvatures.get(sortedCurvatures.size() - 1 - config.getSignificantPointCount());   
        
        List<Integer> significantPoints = new ArrayList<>();
        List<Double> significantCurvatures = new ArrayList<>();
        
        for (int i = 0; i < facet.getNumberOfVertices(); i++) {
            if (curvatures[i] >= bottomCurvature) {
                significantCurvatures.add(curvatures[i]);
                significantPoints.add(i);
            }
        }
        
        Plane plane = new Plane(0, 0, 0, 0);
        int lastVotes = 0;
        
        ProgressMonitor progressMonitor = null;
        if (panel != null) {
            UIManager.put("ProgressMonitor.progressText", "Counting symmetry...");
            progressMonitor = new ProgressMonitor(panel, "Counting...", "Steps", 0, significantPoints.size());
        }
        
        double onePercent = significantCurvatures.size() / 100.0;
        double percentsPerStep = 1 / onePercent;
        for (int i = 0; i < significantPoints.size(); i++) {
            for (int j = 0; j < significantPoints.size(); j++) {
                if (i != j) {
                    double minRatio = config.getMinCurvRatio();
                    double maxRatio = 1.0 / minRatio;
                    if (significantCurvatures.get(i) / significantCurvatures.get(j) >= minRatio && 
                            significantCurvatures.get(i) / significantCurvatures.get(j) <= maxRatio) {
                        
                        Vector3d p1 = new Vector3d(facet.getVertex(significantPoints.get(i)).getPosition());
                        Vector3d p2 = new Vector3d(facet.getVertex(significantPoints.get(j)).getPosition());
                        
                        Vector3d avrg = new Vector3d(p1);
                        avrg.add(p2);
                        avrg.scale(0.5);
                        
                        Vector3d normal = new Vector3d(p1);
                        normal.sub(p2);
                        normal.normalize();
                        
                        double d = -(normal.x * avrg.x) - (normal.y * avrg.y) - (normal.z * avrg.z);
                       
                        Vector3d ni = new Vector3d(facet.getVertex(significantPoints.get(i)).getNormal());
                        Vector3d nj = new Vector3d(facet.getVertex(significantPoints.get(j)).getNormal());
                        ni.normalize();
                        nj.normalize();

                        Vector3d normVec = ni;
                        normVec.sub(nj);
                        normVec.normalize();
                        double normCos = normVec.dot(normal);

                        if (Math.abs(normCos) >= config.getMinNormAngleCos()) {
                            Plane newPlane = new Plane(normal.x, normal.y, normal.z, d);
                            int currentVotes = getVotes(newPlane, 
                                significantCurvatures, 
                                significantPoints, 
                                config.getMinCurvRatio(), 
                                config.getMinAngleCos(), 
                                config.getMinNormAngleCos(), 
                                getBoundingBox().getMaxDiag() * config.getMaxRelDistance());

                            planes.add(new ApproxSymmetryPlane(newPlane, currentVotes));
                            
                            if (currentVotes > lastVotes) {
                                lastVotes = currentVotes;
                                plane = newPlane;
                            }
                        }
                        
                    }
                }
            }
            if (panel != null) {
                progressMonitor.setNote("Task step: " + (int) ((i + 1) * percentsPerStep));
                progressMonitor.setProgress(i);
            }
        }
        
        Collections.sort(planes);
        ArrayList<ApproxSymmetryPlane> finalPlanes = new ArrayList<>();
        for (int i = 0; i < planes.size(); i++) {
            if (planes.get(i).getVotes() == lastVotes){
                finalPlanes.add(planes.get(i));
            }
        }
        Plane finalPlane = computeNewPlane(finalPlanes);
        if (config.isAveraging()){
            plane = finalPlane;
        }
        if (panel != null) {
            JOptionPane.showMessageDialog(panel, "Symmetry estimate done.", "Done", 0,
                   new ImageIcon(getClass().getResource("/cz/fidentis/analyst/gui/resources/exportedModel.png")));
            progressMonitor.close();
        }
        return plane;
    }
    
    private double[] initCurvatures() {
        double[] curvatures = new double[facet.getNumberOfVertices()];
        for (int i = 0; i < facet.getNumberOfVertices(); i++) {
            if (facet.getNumberOfVertices() == 2500) {
                curvatures[i] = this.getMaxCurvature(i);
            } else {
                curvatures[i] = this.getGaussianCurvature(i);
            }
            if (Double.isNaN(curvatures[i])){
                curvatures[i] = Double.MIN_VALUE;
            }
        }
        return curvatures;
    }
    
    private Plane computeNewPlane(List<ApproxSymmetryPlane> finalPlanes) {
        double newA = 0, newB = 0, newC = 0, newD = 0;
        Vector3d refDir = finalPlanes.get(0).getNormal();
        for (int i = 0; i < finalPlanes.size(); i++) {
            Vector3d normDir = finalPlanes.get(i).getNormal();
            if (normDir.dot(refDir) < 0) {
                newA -= normDir.x;
                newB -= normDir.y;
                newC -= normDir.z;
                newD -= finalPlanes.get(i).getDistance();
            } else {
                newA += normDir.x;
                newB += normDir.y;
                newC += normDir.z;
                newD += finalPlanes.get(i).getDistance();
            }
        }
        Plane finalPlane = new Plane(newA, newB, newC, newD);
        finalPlane.normalize();
        return finalPlane;
    }
    
    /**
     * 
     * @param plane Plane computed as symmetry plane
     * @return mesh that represents facet with computed plane of approximate symmetry
     */
    public SymmetryEstimator mergeWithPlane(Plane plane) {
        Vector3d normal = plane.getNormal();
        Vector3d midPoint = getBoundingBox().getMidPoint().getPosition();

        double alpha = -((normal.x * midPoint.x) + 
                (normal.y * midPoint.y) + (normal.z * midPoint.z) +
                plane.getDistance()) / (normal.dot(normal));
        
        Vector3d midPointOnPlane = new Vector3d(midPoint);
        Vector3d nn = new Vector3d(normal);
        nn.scale(alpha);
        midPointOnPlane.add(nn);

        double val = normal.x * midPointOnPlane.x + normal.y *
                midPointOnPlane.y + normal.z *
                midPointOnPlane.z + plane.getDistance();

        Vector3d a = new Vector3d();
        if (Math.abs(normal.dot(new Vector3d(0.0, 1.0, 0.0))) > Math.abs(normal.dot(new Vector3d (1.0, 0.0, 0.0)))) {
            a.cross(normal, new Vector3d(1.0, 0.0, 0.0));
        } else {
            a.cross(normal, new Vector3d(0.0, 1.0, 0.0));
        }
        a.normalize();

        Vector3d b = new Vector3d();
        b.cross(normal,a);
        b.normalize();
        
        SymmetryEstimator planeMesh = new SymmetryEstimator(midPointOnPlane, a, b,
                (getBoundingBox().getMaxPoint().subtractPosition(getBoundingBox().getMinPoint())).getPosition().x);
       
        return mergeMeshWith(planeMesh);
    }
    
    /**
     * 
     * @param s mesh that will be merged
     * @return mesh with merged vertices from both meshes 
     */
    public SymmetryEstimator mergeMeshWith(SymmetryEstimator s) {
        CornerTableRow row1 = new CornerTableRow(facet.getNumberOfVertices(), -1);
        CornerTableRow row2 = new CornerTableRow(facet.getNumberOfVertices() + 1, facet.getNumberOfVertices() + 3);
        CornerTableRow row3 = new CornerTableRow(facet.getNumberOfVertices() + 2, -1);
        CornerTableRow row4 = new CornerTableRow(facet.getNumberOfVertices() + 2, -1);
        CornerTableRow row5 = new CornerTableRow(facet.getNumberOfVertices() + 3, facet.getNumberOfVertices() + 1);
        CornerTableRow row6 = new CornerTableRow(facet.getNumberOfVertices(), -1);
        
        facet.getCornerTable().addRow(row1);
        facet.getCornerTable().addRow(row2);
        facet.getCornerTable().addRow(row3);
        facet.getCornerTable().addRow(row4);
        facet.getCornerTable().addRow(row5);
        facet.getCornerTable().addRow(row6);

        for(int n = 0; n < 4; n++) {
            facet.addVertex(s.getFacet().getVertices().get(n));
        }        
        return this;
    }

    private static MeshFacet createMeshFacet(Vector3d centroid, Vector3d a, Vector3d b, double scale) {
        Vector3d[] points = new Vector3d[4];
        
        Vector3d aScaled = new Vector3d(a);
        Vector3d bScaled = new Vector3d(b);
        aScaled.scale(scale);
        bScaled.scale(scale);
        
        points[0] = new Vector3d(centroid);
        points[0].sub(aScaled);
        points[0].sub(bScaled);
        
        points[1] = new Vector3d(centroid);
        points[1].sub(aScaled);
        points[1].add(bScaled);
        
        points[2] = new Vector3d(centroid);
        points[2].add(aScaled);
        points[2].add(bScaled);
        
        points[3] = new Vector3d(centroid);
        points[3].add(aScaled);
        points[3].sub(bScaled);
        
        MeshFacet facet = new MeshFacetImpl();
        for (Vector3d point : points) {
            facet.addVertex(new MeshPointImpl(point, null, null));
        }
        facet.calculateVertexNormals();
        
        return facet;
    }

    /**
     * Representation of areas of Voronoi region of triangles
     * used for computing Gaussian curvature 
     * 
     * @author Natália Bebjaková
     */
    private final class TriangleVertexAreas {
            public final double v1Area;
            public final double v2Area;
            public final double v3Area;
            
            private TriangleVertexAreas(double v1, double v2, double v3) {
                v1Area = v1;
                v2Area = v2;
                v3Area = v3;
            }
    }
   
    /**
     * 
     * @param pointIndex index of point for which we are searching traingles in its neighborhood
     * @return triangle neighbours of given index of vertex
     */
    protected List<Integer> getNeighbours(int pointIndex) {
        return facet.getCornerTable().getTriangleIndexesByVertexIndex(pointIndex);
    }
    
    /**
     * Return area of Voronoi region of triangle
     * 
     * @param t triangle of which area is computed
     * @return area of Voronoi region
     */
    private TriangleVertexAreas computeTriangleVertexAreas(MeshTriangle t) {
        double a = (t.vertex2.subtractPosition(t.vertex3)).abs();
        double b = (t.vertex3.subtractPosition(t.vertex1)).abs();
        double c = (t.vertex2.subtractPosition(t.vertex1)).abs();
        
        double d1 = a * a * (b * b + c * c - a * a);
        double d2 = b * b * (c * c + a * a - b * b);
        double d3 = c * c * (a * a + b * b - c * c);
        double dSum = d1 + d2 + d3;
        
        d1 /= dSum;
        d2 /= dSum;
        d3 /= dSum;

        MeshPoint v1Half = (t.vertex2.addPosition(t.vertex3)).dividePosition(2);
        MeshPoint v2Half = (t.vertex1.addPosition(t.vertex3)).dividePosition(2);
        MeshPoint v3Half = (t.vertex2.addPosition(t.vertex1)).dividePosition(2);

        
        //TriangleVertexAreas area = new TriangleVertexAreas();

        if (d1 < 0) {           
            double v3Area = ((v2Half.subtractPosition(t.vertex3)).crossProduct(v1Half.subtractPosition(t.vertex3))).abs() / 2.0;
            double v2Area = ((v3Half.subtractPosition(t.vertex2)).crossProduct(v1Half.subtractPosition(t.vertex2))).abs() / 2.0;
            double v1Area = (((v1Half.subtractPosition(t.vertex1)).crossProduct(v3Half.subtractPosition(t.vertex1))).abs() / 2.0) + 
                    (((v1Half.subtractPosition(t.vertex1)).crossProduct(v2Half.subtractPosition(t.vertex1))).abs() / 2.0);
            return new TriangleVertexAreas(v1Area, v2Area, v3Area);
        }
        if (d2 < 0) {
            double v1Area = ((v3Half.subtractPosition(t.vertex1)).crossProduct(v2Half.subtractPosition(t.vertex1))).abs() / 2.0;
            double v3Area = ((v1Half.subtractPosition(t.vertex3)).crossProduct(v2Half.subtractPosition(t.vertex3))).abs() / 2.0;
            double v2Area = (((v2Half.subtractPosition(t.vertex2)).crossProduct(v1Half.subtractPosition(t.vertex2))).abs() / 2.0) + 
                    (((v2Half.subtractPosition(t.vertex2)).crossProduct(v3Half.subtractPosition(t.vertex2))).abs() / 2.0);
            return new TriangleVertexAreas(v1Area, v2Area, v3Area);
        }
        if (d3 < 0) {
            double v2Area = ((v1Half.subtractPosition(t.vertex2)).crossProduct(v3Half.subtractPosition(t.vertex2))).abs() / 2.0;
            double v1Area = ((v2Half.subtractPosition(t.vertex1)).crossProduct(v3Half.subtractPosition(t.vertex1))).abs() / 2.0;
            double v3Area = (((v3Half.subtractPosition(t.vertex3)).crossProduct(v2Half.subtractPosition(t.vertex3))).abs() / 2.0) + 
                    (((v3Half.subtractPosition(t.vertex3)).crossProduct(v1Half.subtractPosition(t.vertex3))).abs() / 2.0);
            return new TriangleVertexAreas(v1Area, v2Area, v3Area);
        }
        
        MeshPoint circumcenter = t.vertex1.multiplyPosition(d1).addPosition(t.vertex2.multiplyPosition(d2).addPosition(t.vertex3.multiplyPosition(d3)));
        
        double v1Area = (((v2Half.subtractPosition(t.vertex1)).crossProduct(circumcenter.subtractPosition(t.vertex1))).abs() / 2.0) + 
                (((v3Half.subtractPosition(t.vertex1)).crossProduct(circumcenter.subtractPosition(t.vertex1))).abs() / 2.0);
        
        double v2Area = (((v3Half.subtractPosition(t.vertex2)).crossProduct(circumcenter.subtractPosition(t.vertex2))).abs() / 2.0) +
                (((v1Half.subtractPosition(t.vertex2)).crossProduct(circumcenter.subtractPosition(t.vertex2))).abs() / 2.0);
        
        double v3Area = (((v1Half.subtractPosition(t.vertex3)).crossProduct(circumcenter.subtractPosition(t.vertex3))).abs() / 2.0) +
                (((v2Half.subtractPosition(t.vertex3)).crossProduct(circumcenter.subtractPosition(t.vertex3))).abs() / 2.0);
        return new TriangleVertexAreas(v1Area, v2Area, v3Area);
    }
    
    /**
     * Calculates angle of the triangle according to center point
     * 
     * @param centerPoint center point
     * @param t triangle
     * @return angle
     */
    private double calculateTriangleAlfa(MeshPoint centerPoint, MeshTriangle t) {        
        double alfaCos = 0;
        
        if (t.vertex1 == centerPoint) {
            MeshPoint e1 = t.vertex1.subtractPosition(t.vertex3);
            e1 = e1.dividePosition(e1.abs());
            
            MeshPoint e2 = t.vertex2.subtractPosition(t.vertex3);
            e2 = e2.dividePosition(e2.abs());
            
            alfaCos = e1.dotProduct(e2);
        } else if (t.vertex2 == centerPoint) {
            MeshPoint e1 = t.vertex2.subtractPosition(t.vertex1);
            e1 = e1.dividePosition(e1.abs());
            
            MeshPoint e2 = t.vertex3.subtractPosition(t.vertex1);
            e2 = e2.dividePosition(e2.abs());
            
            alfaCos = e1.dotProduct(e2);
        } else if (t.vertex3 == centerPoint){
            MeshPoint e1 = t.vertex3.subtractPosition(t.vertex2);
            e1 = e1.dividePosition(e1.abs());
                
            MeshPoint e2 = t.vertex1.subtractPosition(t.vertex2);
            e2 = e2.dividePosition(e2.abs());
            
            alfaCos = e1.dotProduct(e2);
        }
        double alfa = Math.acos(alfaCos);
        return 1 / Math.tan(alfa);
    }

    /**
     * Calculates angle of the triangle according to center point
     * 
     * @param centerPoint center point
     * @param t triangle
     * @return angle
     */
    private double calculateTriangleBeta(MeshPoint centerPoint, MeshTriangle t) {
        double betaCos = 0;
            
        if (t.vertex1 == centerPoint) {
            MeshPoint e1 = t.vertex1.subtractPosition(t.vertex2);
            e1 = e1.dividePosition(e1.abs());
            
            MeshPoint e2 = t.vertex3.subtractPosition(t.vertex2);
            e2 = e2.dividePosition(e2.abs());
            
            betaCos = e1.dotProduct(e2);
        } else if (t.vertex2 == centerPoint) {
            MeshPoint e1 = t.vertex2.subtractPosition(t.vertex3);
            e1 = e1.dividePosition(e1.abs());
            
            MeshPoint e2 = t.vertex1.subtractPosition(t.vertex3);
            e2 = e2.dividePosition(e2.abs());
            
            betaCos = e1.dotProduct(e2);
        } else if (t.vertex3 == centerPoint){
            MeshPoint e1 = t.vertex3.subtractPosition(t.vertex1);
            e1 = e1.dividePosition(e1.abs());
                
            MeshPoint e2 = t.vertex2.subtractPosition(t.vertex1);
            e2 = e2.dividePosition(e2.abs());
            
            betaCos = e1.dotProduct(e2);
        }
            double beta = Math.acos(betaCos);
            return 1 / Math.tan(beta);
    }
        
    /**
     * Calculates Laplacian 
     * 
     * @param centerIndex centerIndex
     * @return laplacian 
     */
    private Vector3d calculateLaplacian(int centerIndex) {
        List<Integer> trianglesNeighbours = getNeighbours(centerIndex);

        if (trianglesNeighbours.isEmpty()) {
            return new Vector3d();
        }
        
        double areaSum = 0;
        Vector3d pointSum = new Vector3d();
        for (int i = 0; i < trianglesNeighbours.size(); i++) {
            MeshTriangle t = triangles.get(trianglesNeighbours.get(i));
            MeshTriangle tNext = triangles.get(trianglesNeighbours.get((i + 1) % trianglesNeighbours.size()));
            double area = 0;
            
            //MeshPoint tVertex1 = facet.getVertices().get(t.vertex1);
            //MeshPoint tVertex2 = facet.getVertices().get(t.vertex2);
            //MeshPoint tVertex3 = facet.getVertices().get(t.vertex3);
            MeshPoint centerPoint = facet.getVertices().get(centerIndex);

            Vector3d aux = new Vector3d();
            if (t.vertex1 == facet.getVertex(centerIndex)) {
                area = areas[trianglesNeighbours.get(i)].v1Area;
                aux = new Vector3d(t.vertex2.getPosition());
            } else if (t.vertex2 == facet.getVertex(centerIndex)) {
                area = areas[trianglesNeighbours.get(i)].v2Area;
                aux = new Vector3d(t.vertex3.getPosition());
            } else if (t.vertex3 == facet.getVertex(centerIndex)) {
                area = areas[trianglesNeighbours.get(i)].v3Area;
                aux = new Vector3d(t.vertex1.getPosition());
            }
            aux.sub(centerPoint.getPosition());
            aux.scale((calculateTriangleAlfa(centerPoint, t) + calculateTriangleBeta(centerPoint, tNext)) / 2.0);
            pointSum.add(aux);
            areaSum += area;
        }
        pointSum.scale(1.0/areaSum);
        return pointSum;
    }
    
    /**
     * Gaussian curvature in a vertex of a triangle mesh.
     * It can only be estimated because triangle mesh is not a continuous
     * but discrete representation of surface.
     * 
     * @param centerIndex index of vertex in which gaussian curvature is computed
     * @return Gaussian curvature in given vertex
     */
    private double getGaussianCurvature(int centerIndex) {
        
        List<Integer> triangleNeighbours = this.getNeighbours(centerIndex);
        
        if (triangleNeighbours.isEmpty()) {
            return Double.NaN;
        }
        double sum = 2 * Math.PI;
        double areaSum = 0;

        for (int i = 0; i < triangleNeighbours.size(); i++) {
            Vector3d v1 = new Vector3d();
            Vector3d v2 = new Vector3d();
            MeshTriangle t = triangles.get(triangleNeighbours.get(i));
            
            double area = 0;
            if (t.vertex1 ==  facet.getVertex(centerIndex)) {
                v1 = new Vector3d(t.vertex2.getPosition());
                v2 = new Vector3d(t.vertex3.getPosition());
                v1.sub(t.vertex1.getPosition());
                v2.sub(t.vertex1.getPosition());

                area = areas[triangleNeighbours.get(i)].v1Area;
            } else if (t.vertex2 == facet.getVertex(centerIndex)) {
                v1 = new Vector3d(t.vertex1.getPosition());
                v2 = new Vector3d(t.vertex3.getPosition());
                v1.sub(t.vertex2.getPosition());
                v2.sub(t.vertex2.getPosition());
                
                area = areas[triangleNeighbours.get(i)].v2Area;
            } else if (t.vertex3 == facet.getVertex(centerIndex)) {
                v1 = new Vector3d(t.vertex2.getPosition());
                v2 = new Vector3d(t.vertex1.getPosition());
                v1.sub(t.vertex3.getPosition());
                v2.sub(t.vertex3.getPosition());

                area = areas[triangleNeighbours.get(i)].v3Area;
            }
            
            areaSum += area;
            v1.normalize();
            v2.normalize();
            
            sum -= Math.acos(v1.dot(v2));
        } 
        double value = sum * (1.0 / areaSum);
        return value;
    }

    /**
     * 
     * @param centerIndex center index
     * @return mean curvature
     */
    private double getMeanCurvature(int centerIndex) {
        return calculateLaplacian(centerIndex).length();
    }
    
    /**
     * 
     * @param centerIndex center index
     * @return max curvature
     */
    private double getMaxCurvature(int centerIndex) {
        double gaussianCurvature = getGaussianCurvature(centerIndex);
        if (Double.isNaN(gaussianCurvature)) {
            return Double.NaN;
        }
        double meanCurvature = getMeanCurvature(centerIndex);
        double meanCurvatureSqr = meanCurvature * meanCurvature;
        if (meanCurvatureSqr <= gaussianCurvature) { 
            return meanCurvature;
        }
        return meanCurvature + Math.sqrt(meanCurvatureSqr - gaussianCurvature);
    }
    
    /**
     * Computes votes for given plane 
     * 
     * @param plane Plane for which votes are computed
     * @param curvatures significant curvatures chosen for computing
     * @param points significant points chosen for computing
     * @param minCurvRatio optional parameter from configuration
     * @param minAngleCos optional parameter from configuration
     * @param minNormAngleCos optional parameter from configuration
     * @param maxDist optional parameter from configuration
     * @return total votes given to plane while computing the symmetry
     */
    private int getVotes(Plane plane,
            List<Double> curvatures,
            List<Integer> points,
            double minCurvRatio,
            double minAngleCos,
            double minNormAngleCos,
            double maxDist) {
        plane.normalize();
        
        Vector3d normal = plane.getNormal();
        double d = plane.getDistance();
        double maxCurvRatio = 1.0 / minCurvRatio;
        int votes = 0;
        //List<Vector3d> normals = calculateNormals();
        
        if (!facet.hasVertexNormals()) {
            facet.calculateVertexNormals();
        }

        for (int i = 0; i < curvatures.size(); i++) {
            for (int j = 0; j < curvatures.size(); j++) {
                if (i != j && (curvatures.get(i) / curvatures.get(j) >= minCurvRatio
                        && curvatures.get(i) / curvatures.get(j) <= maxCurvRatio)) {
                    
                    Vector3d vec = new Vector3d(facet.getVertices().get(points.get(i)).getPosition());
                    vec.sub(facet.getVertices().get(points.get(j)).getPosition());
                    vec.normalize();
                    double cos = vec.dot(normal);
                    
                    Vector3d ni = new Vector3d(facet.getVertex(points.get(i)).getNormal());
                    Vector3d nj = new Vector3d(facet.getVertex(points.get(j)).getNormal());
                    ni.normalize();
                    nj.normalize();
                    
                    Vector3d normVec = ni;
                    normVec.sub(nj);
                    normVec.normalize();
                    double normCos = normVec.dot(normal);
 
                    if (Math.abs(cos) >= minAngleCos && Math.abs(normCos) >= minNormAngleCos) {
                        Vector3d avrg = new Vector3d(facet.getVertices().get(points.get(i)).getPosition());
                        Vector3d aux = new Vector3d(facet.getVertices().get(points.get(i)).getPosition());
                        avrg.add(aux);
                        avrg.scale(0.5);
                        
                        double dist = normal.x * avrg.x + normal.y * avrg.y + normal.z * avrg.z + d;
                        
                        dist = Math.abs(dist);
                        
                        if (dist <= maxDist) {                 
                            votes++;  
                        }   
                    }
                }
            }
        }
        return votes;
    }
    
}
