package cz.fidentis.analyst.visitors.mesh;

import cz.fidentis.analyst.mesh.MeshVisitor;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import cz.fidentis.analyst.mesh.core.MeshPointImpl;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.vecmath.Vector3d;

/**
 * This visitor finds mesh points that are the closest to the given 3D point.
 * Because there can be multiple points with the same minimal distance, the visitor
 * returns a list of mesh facets and a list of indices corresponding 
 * to the facets. 
 * <p>
 * Returned lists have the same size. 
 * {@code n = getIndices().get(i)} returns the index of the closest point on the i-th facet.
 * Then call {@code getClosestFacets().get(i).getVertex(n))} to get the closest point.
 * </p>
* <p>
 * This visitor is thread-safe. A single instance of the visitor can be used 
 * to inspect multiple mesh models or facets simultaneously.
 * </p>
  * 
 * @author Matej Lukes
 * @author Radek Oslejsek
 */
public class Point2MeshVisitor extends MeshVisitor {
    
    private boolean relativeDist;
    private MeshPoint myPoint;
    private double distance = Double.POSITIVE_INFINITY;
    private int sign = 1;
    private final List<MeshFacet> closestFacets = new ArrayList<>();
    private final List<Integer> indices = new ArrayList<>();
    
    /**
     * @param point Mesh point of which distance is computed. Must not be {@code null}
     * @param relativeDistance If true, the visitor calculates the distances with respect 
     * to the normals of the source point (the normal has to be present), 
     * i.e., we can get negative distance.
     * @param concurrently If {@code true} and this visitor is thread-safe, then
     * the visitor is applied concurrently on multiple mesf facets.
     * @throws IllegalArgumentException if some parametr is wrong
     */
    public Point2MeshVisitor(MeshPoint point, boolean relativeDistance, boolean concurrently) {
        super(concurrently);
        if (point == null) {
            throw new IllegalArgumentException("point");
        }
        if (point.getNormal() == null && relativeDistance) {
            throw new IllegalArgumentException("normal required for relative distance");
        }
        this.myPoint = point;
        this.relativeDist = relativeDistance;
    }
    
    /**
     * @param point 3D point of which distance is computed. Must not be {@code null}
     * @param concurrently If {@code true} and this visitor is thread-safe, then
     * the visitor is applied concurrently on multiple mesf facets.
     * @throws IllegalArgumentException if some parametr is wrong
     */
    public Point2MeshVisitor(Vector3d point, boolean concurrently) {
        this (new MeshPointImpl(point, null, null), true, concurrently);
    }

    @Override
    protected void visitMeshFacet(MeshFacet facet) {
        List<MeshPoint> vertices = facet.getVertices();
        for (int i = 0; i < vertices.size(); i++) {
            checkAndUpdateDistance(vertices.get(i).getPosition(), facet, i);
        }
    }
    
    /**
     * Returns distance to the closest mesh point.
     * 
     * @return distance to the closest mesh point, 
     * {@code Double.POSITIVE_INFINITY} if no distance has been calculated
     */
    public double getDistance() {
        return relativeDist ? sign * distance : distance;
    }
    
    /**
     * Returns indeces to the closest elements (points or triangles) of corresponding facet.
     * 
     * @return indeces to the closest elements (points or triangles) of corresponding facet,
     * {@code null} if no search has been performed
     */
    public List<Integer> getIndices() {
        return Collections.unmodifiableList(indices);
    }
    
    /**
     * Returns the closest mesh facets (containing the closest mesh points).
     * 
     * @return the closest mesh facets
     */
    public List<MeshFacet> getClosestFacets() {
        return Collections.unmodifiableList(closestFacets);
    }

    protected boolean isRelativeDist() {
        return relativeDist;
    }

    protected MeshPoint getMyPoint() {
        return myPoint;
    }

    protected int getSign() {
        return sign;
    }

    protected void checkAndUpdateDistance(Vector3d other, MeshFacet facet, int index) {
        Vector3d aux = new Vector3d(other);
        aux.sub(myPoint.getPosition());
        sign = relativeDist ? (int) Math.signum(aux.dot(myPoint.getNormal())): 1;
        double dist = aux.length();
            
        synchronized (this) {
            if (dist > distance) {
                return;
            }
        
            if (dist < distance) { // new closest point
                distance = dist;
                closestFacets.clear();
                indices.clear();
                closestFacets.add(facet);
                indices.add(index);
            } else { // the same distance
                closestFacets.add(facet);
                indices.add(index);
            }
        }
    }
}
