package cz.fidentis.analyst.visitors.mesh;

import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import cz.fidentis.analyst.mesh.core.MeshPointImpl;
import cz.fidentis.analyst.mesh.core.MeshTriangle;
import java.util.ArrayList;
import java.util.List;
import javax.vecmath.Vector3d;

/**
 * This visitor finds mesh points that are the closest to the given 3D point.
 * In contrast to the {@code Point2MeshVisitor} visitor, 
 * which computes pont-to-point distance, this implementation measures 
 * the distance to the triangles. 
 * <p>
 * Because there can be multiple triangles with the same minimal distance, the visitor
 * returns a list of mesh facets and a list of indices corresponding 
 * to the facets.  
 * </p>
 * <p>
 * Returned lists have the same size. 
 * {@code n = getIndices().get(i)} returns the index of the closest triangle on the i-th facet.
 * Then iterate into the n-th triangle of facet {@code getClosestFacets().get(i)} to get 
 * the triangle instance.
 * </p>
* <p>
 * This visitor is thread-safe. A single instance of the visitor can be used 
 * to inspect multiple mesh models or facets simultaneously.
 * </p>
  * 
 * @author Matej Lukes
 * @author Radek Oslejsek
 */
public class Point2MeshTriVisitor extends Point2MeshVisitor {
    
    public static final double EPS = 0.001; // tollerance for double computations
    
    /**
     * @param point Mesh point of which distance is computed. Must not be {@code null}
     * @param relativeDistance If true, the visitor calculates the distances with respect 
     * to the normals of the source point (the normal has to be present), 
     * i.e., we can get negative distance.
     * @param concurrently If {@code true} and this visitor is thread-safe, then
     * the visitor is applied concurrently on multiple mesf facets.
     * @throws IllegalArgumentException if some parametr is wrong
     */
    public Point2MeshTriVisitor(MeshPoint point, boolean relativeDistance, boolean concurrently) {
        super(point, relativeDistance, concurrently);
    }
    
    /**
     * @param point 3D point of which distance is computed. Must not be {@code null}
     * @param concurrently If {@code true} and this visitor is thread-safe, then
     * the visitor is applied concurrently on multiple mesf facets.
     * @throws IllegalArgumentException if some parametr is wrong
     */
    public Point2MeshTriVisitor(Vector3d point, boolean concurrently) {
        this (new MeshPointImpl(point, null, null), true, concurrently);
    }
    
    @Override
    protected void visitMeshFacet(MeshFacet facet) {
        Vector3d my = getMyPoint().getPosition();
        int i = 0;
        for (MeshTriangle tri: facet) {
            Vector3d projection = getProjectionToTrianglePlane(my, tri);
            if (!isPointInTriangle(projection, tri)) {
                projection = getProjectionToClosestEdge(projection, tri);
            }
            checkAndUpdateDistance(projection, facet, i);
        }        
    }
    
    /**
     * returns perpendicular projection from vertex to plane of triangle
     *
     * @param vertex   vertex from which the projection is created
     * @param triangle triangle that defines the plane
     * @return projection to plane of triangle
     */
    private Vector3d getProjectionToTrianglePlane(Vector3d vertex, MeshTriangle triangle) {
        Vector3d ab = new Vector3d(triangle.vertex1.getPosition());
        ab.sub(triangle.vertex2.getPosition());
        Vector3d ac = new Vector3d(triangle.vertex1.getPosition());
        ac.sub(triangle.vertex3.getPosition());
        Vector3d normal = new Vector3d();
        normal.cross(ab, ac);
        normal.normalize();

        Vector3d projection = new Vector3d(vertex);
        Vector3d helperVector = new Vector3d(vertex);
        helperVector.sub(triangle.vertex1.getPosition());
        double dist = helperVector.dot(normal);

        projection.scaleAdd(-dist, normal, projection);
        return projection;
    }

    /**
     * checks if a point in the plane of triangle lies within the triangle
     *
     * @param point    checked point
     * @param triangle triangle
     * @return true if point is in triangle, false otherwise
     */
    private boolean isPointInTriangle(Vector3d point, MeshTriangle triangle) {
        List<Vector3d> pointToVertices = new ArrayList<>();
        for (MeshPoint p: triangle) {
            Vector3d v = new Vector3d(p.getPosition());
            v.sub(point);
            pointToVertices.add(v);
        }
        
        double angleSum = 0;
        for (int i = 0; i < 3; i++) {
            angleSum += pointToVertices.get(i).angle(pointToVertices.get((i + 1) % 3));
        }
        angleSum -= 2 * Math.PI;
        return -EPS < angleSum && angleSum < EPS;
    }

    /**
     * returns projection to the nearest edge of the triangle
     *
     * @param point    point in plane of triangle
     * @param triangle triangle
     * @return perpendicular projection to the nearest edge
     */
    private Vector3d getProjectionToClosestEdge(Vector3d point, MeshTriangle triangle) {
        Vector3d[] projections = new Vector3d[3];
        projections[0] = getProjectionToEdge(point, triangle.vertex1.getPosition(), triangle.vertex2.getPosition());
        projections[1] = getProjectionToEdge(point, triangle.vertex2.getPosition(), triangle.vertex3.getPosition());
        projections[2] = getProjectionToEdge(point, triangle.vertex3.getPosition(), triangle.vertex1.getPosition());
        
        double minDistance = Double.MAX_VALUE;
        Vector3d closestProjection = null;
        Vector3d helperVector = new Vector3d();
        for (Vector3d projection: projections) {
            helperVector.sub(point, projection);
            double dist = helperVector.length();
            if (dist < minDistance) {
                minDistance = dist;
                closestProjection = projection;
            }
        }
        return closestProjection;
    }

    /**
     * returns projection to edge
     *
     * @param point       point in plane of triangle
     * @param edgeVertex1 first vertex of edge
     * @param edgeVertex2 second vertex of edge
     * @return projection to edge
     */
    private Vector3d getProjectionToEdge(Vector3d point, Vector3d edgeVertex1, Vector3d edgeVertex2) {
        Vector3d ab = new Vector3d();
        ab.sub(edgeVertex2, edgeVertex1);
        Vector3d ap = new Vector3d();
        ap.sub(point, edgeVertex1);
        double t = ab.dot(ap) / ab.lengthSquared();
        Vector3d projection = new Vector3d(
                edgeVertex1.x + t * ab.x,
                edgeVertex1.y + t * ab.y,
                edgeVertex1.z + t * ab.z);

        Vector3d projectionToEdgeVertex1 = new Vector3d();
        projectionToEdgeVertex1.sub(edgeVertex1, projection);
        Vector3d projectionToEdgeVertex2 = new Vector3d();
        projectionToEdgeVertex2.sub(edgeVertex2, projection);

        if (Math.abs((projectionToEdgeVertex1.length() + projectionToEdgeVertex2.length()) - ab.length()) < EPS) {
            return projection;
        }

        if (projectionToEdgeVertex1.length() < projectionToEdgeVertex2.length()) {
            return edgeVertex1;
        }

        return edgeVertex2;
    }

}
