package cz.fidentis.analyst.visitors.mesh;

import cz.fidentis.analyst.mesh.MeshVisitor;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Visitor for Hausdorff distance. 
 * This visitor is instantiated on a single mesh facet or multiple facets. 
 * When applied to other facets, it computes Huasdorff distance to them.
 * 
 * @author Matej Lukes
 * @author Radek Oslejsek
 */
public class HausdorffDistMeshVisitor extends MeshVisitor {
    
    private boolean relativeDistance;
    private Map<MeshFacet, List<Double>> distances = new HashMap<>();
    
    /**
     * @param mainFacets Facets whose distance to other facets is computed. Must not be {@code null}
     * @param relativeDistance If true, then the visitor calculates the relative distances with respect 
     * to the normal vectors of source facets (normal vectors have to present), 
     * i.e., we can get negative distances.
     * @param concurrently If {@code true} and this visitor is thread-safe, then
     * the visitor is applied concurrently on multiple mesf facets.
     * @throws IllegalArgumentException if some parametr is wrong
     */
    public HausdorffDistMeshVisitor(Set<MeshFacet> mainFacets, boolean relativeDistance, boolean concurrently) {
        super(concurrently);
        if (mainFacets == null) {
            throw new IllegalArgumentException("mainFacets");
        }
        for (MeshFacet f: mainFacets) {
            distances.put(f, new ArrayList<>());
        }
        this.relativeDistance = relativeDistance;
    }
    
    /**
     * @param mainFacet Primary facet of which distance to others is to be computed. Must not be {@code null}
     * @param relativeDistance If true, then the visitor calculates the relative distances with respect 
     * to the normal vectors of source facets (normal vectors have to present), 
     * i.e., we can get negative distances.
     * @param concurrently If {@code true} and this visitor is thread-safe, then
     * the visitor is applied concurrently on multiple mesf facets.
     * @throws IllegalArgumentException if some parametr is wrong
     */
    public HausdorffDistMeshVisitor(MeshFacet mainFacet, boolean relativeDistance, boolean concurrently) {
        this(new HashSet<>(Collections.singleton(mainFacet)), relativeDistance, concurrently);
        if (mainFacet == null) {
            throw new IllegalArgumentException("mainFacet");
        }
    }
    
    @Override
    protected void visitMeshFacet(MeshFacet comparedFacet) {
        for (Map.Entry<MeshFacet, List<Double>> entry: distances.entrySet()) {
            List<MeshPoint> vertices = entry.getKey().getVertices();
            List<Double> distList = entry.getValue();
            
            boolean firstComparison = distList.isEmpty();
            
            for (int i = 0; i < vertices.size(); i++) {
                Point2MeshVisitor visitor = new Point2MeshVisitor(vertices.get(i), relativeDistance, concurrently());
                comparedFacet.accept(visitor);
                double dist = visitor.getDistance();
                updateDistances(distList, firstComparison, dist, i);
            }
        }
    }
    
    /**
     * Return Hausdorff distance for all points of the source mesh. 
     * A list of distances is returned for each mesh separaptely. Index of the 
     * list corresponds to the order of vertives in the mesh facet. 
     * Therefore, i-t number corresponds to the distance of i-th vertex of the 
     * source mesh.
     * 
     * @return Hausdorff distance for all points of the source mesh
     */
    public Map<MeshFacet, List<Double>> getDistances() {
        return Collections.unmodifiableMap(distances);
    }
    
    protected boolean isRelativeDistance() {
        return relativeDistance;
    }
    
    protected Map<MeshFacet, List<Double>> getDistMap() {
        return distances;
    }
    
    protected void updateDistances(List<Double> distList, boolean firstComparison, double dist, int index) {
        if (firstComparison) {
            distList.add(dist);
            return;
        } 
        
        if (dist >= 0 && dist < distList.get(index)) {
            distList.set(index, dist);
            return;
        }
        
        if (dist < 0 && dist > distList.get(index)) { // for relative dist only
            distList.set(index, dist);
        }
    }
}
