package cz.fidentis.analyst.visitors.mesh;

import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Visitor for Hausdorff distance. 
 * This visitor is instantiated on a single mesh facet or multiple facets. 
 * When applied to other facets, it computes Huasdorff distance to them.
 * 
 * Distances to triangles are computed, in constrast to the {@code HausdorffDistMeshTriVisitor}
 * which calculated distances to mesh points.
 * 
 * @author Matej Lukes
 * @author Radek Oslejsek
 */
public class HausdorffDistMeshTriVisitor extends HausdorffDistMeshVisitor {
    
    /**
     * @param mainFacets Facets whose distance to other facets is computed. Must not be {@code null}
     * @param relativeDistance If true, then the visitor calculates the relative distances with respect 
     * to the normal vectors of source facets (normal vectors have to present), 
     * i.e., we can get negative distances.
     * @param concurrently If {@code true} and this visitor is thread-safe, then
     * the visitor is applied concurrently on multiple mesf facets.
     * @throws IllegalArgumentException if some parametr is wrong
     */
    public HausdorffDistMeshTriVisitor(Set<MeshFacet> mainFacets, boolean relativeDistance, boolean concurrently) {
        super(mainFacets, relativeDistance, concurrently);
    }
    
    /**
     * @param mainFacet Primary facet of which distance to others is to be computed. Must not be {@code null}
     * @param relativeDistance If true, then the visitor calculates the relative distances with respect 
     * to the normal vectors of source facets (normal vectors have to present), 
     * i.e., we can get negative distances.
     * @param concurrently If {@code true} and this visitor is thread-safe, then
     * the visitor is applied concurrently on multiple mesf facets.
     * @throws IllegalArgumentException if some parametr is wrong
     */
    public HausdorffDistMeshTriVisitor(MeshFacet mainFacet, boolean relativeDistance, boolean concurrently) {
        super(mainFacet, relativeDistance, concurrently);
    }
    
    @Override
    protected void visitMeshFacet(MeshFacet comparedFacet) {
        for (Map.Entry<MeshFacet, List<Double>> entry: getDistMap().entrySet()) {
            List<MeshPoint> vertices = entry.getKey().getVertices();
            List<Double> distList = entry.getValue();
            
            boolean firstComparison = distList.isEmpty();
            
            for (int i = 0; i < vertices.size(); i++) {
                Point2MeshTriVisitor visitor = new Point2MeshTriVisitor(vertices.get(i), isRelativeDistance(), concurrently());
                comparedFacet.accept(visitor);
                double dist = visitor.getDistance();
                updateDistances(distList, firstComparison, dist, i);
            }
        }
    }
}
