package cz.fidentis.analyst;

import cz.fidentis.analyst.face.AvgFaceConstructor;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.face.HumanFaceFactory;
import cz.fidentis.analyst.icp.IcpTransformer;
import cz.fidentis.analyst.icp.UndersamplingStrategy;
import cz.fidentis.analyst.kdtree.KdTree;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.symmetry.SignificantPoints;
import static cz.fidentis.analyst.symmetry.SignificantPoints.CurvatureAlg.GAUSSIAN;
import cz.fidentis.analyst.symmetry.SymmetryConfig;
import cz.fidentis.analyst.symmetry.SymmetryEstimator;
import java.io.File;
import java.util.ArrayList;
import java.util.List;

/**
 * Helper class providing methods for batch N:N processing.
 * 
 * @author Radek Oslejsek
 */
public class BatchProcessor {
    
    /**
     * Template face computed by averaging vertices processed faces.
     */
    private HumanFace templateFace = null;
    
    private double avgIcpIterations = 0.0;
    
    private SymmetryConfig symmetryConfig;
    private SignificantPoints.CurvatureAlg symmetryCurvArg = GAUSSIAN;
    
    /**
     * Constructor.
     * @param symmetryPoints Number of symmetry points used by the symmetry estimator
     */
    public BatchProcessor(int symmetryPoints) {
        symmetryConfig = new SymmetryConfig();
        this.symmetryConfig.setSignificantPointCount(symmetryPoints); // default 200, best 500
    }

    /**
     * Pre-reads faces from given directory into the HumanFactory.
     * @param dir Directory.
     * @return Face IDs
     */
    public List<String> readFaces(String dir) {
        List<String> faces = new ArrayList<>();
        for (File subdir : (new File(dir)).listFiles(File::isDirectory)) {
            if (subdir.getName().matches("^\\d\\d$")) {
                for (File file: subdir.listFiles(File::isFile)) {
                    if (file.getName().endsWith(".obj")) {
                        //System.out.print(file.getName() + " ");
                        //if (file.getName().endsWith("00002_01_ECA.obj")) { // FOT DEBUGING
                            String faceId = HumanFaceFactory.instance().loadFace(file);
                            faces.add(faceId);
                        //}
                    }
                }
            }
        }
        return faces;
    }
    
    /**
     * Pre-computes symmetry planes for given faces.
     * @param faceIDs Face IDs
     */
    public void computeSymmetryPlanes(List<String> faceIDs) {
        for (String facePath: faceIDs) {
            String faceId = HumanFaceFactory.instance().loadFace(new File(facePath));
            HumanFace face = HumanFaceFactory.instance().getFace(faceId);
            SymmetryEstimator se = new SymmetryEstimator(this.symmetryConfig, this.symmetryCurvArg);
            face.getMeshModel().compute(se);
            face.setSymmetryPlane(se.getSymmetryPlane(), se.getSymmetryPlaneMesh());
        }
    }
    
    /**
     * Registers faces with respect to the given primary face.
     * 
     * @param primaryFaceId ID of the primary face
     * @param faceIDs IDs of faces that are to be superimposed (registered)
     * @param iters Maximal number of ICP iterations
     * @param error Maximal error of ICP computation
     * @param strategy Undersampling strategy
     */
    public void superimposeOnPrimaryFace(
            String primaryFaceId, List<String> faceIDs, int iters, double error, UndersamplingStrategy strategy) {
        
        if (faceIDs == null) {
            throw new IllegalArgumentException("faces");
        }
        
        HumanFaceFactory.instance().setStrategy(HumanFaceFactory.Strategy.MRU);
        
        int facesCounter = faceIDs.size();
        HumanFace primaryFace = HumanFaceFactory.instance().getFace(primaryFaceId);
        KdTree primKdTree = primaryFace.computeKdTree(true);
        IcpTransformer icpTransformer = new IcpTransformer(primKdTree, iters, false, error, strategy);
            
        for (String faceId: faceIDs) {
            //String faceId = HumanFaceFactory.instance().getFace(new File(facePath));
            if (primaryFace.getId().equals(faceId)) { // skip the same face
                facesCounter--;
                continue;
            }
            
            HumanFace face = HumanFaceFactory.instance().getFace(faceId);
            face.getMeshModel().compute(icpTransformer);
            face.removeKdTree();
        }
        
        int itersCounter = 0;
        for (MeshFacet f: icpTransformer.getTransformations().keySet()) {
            itersCounter += icpTransformer.getTransformations().get(f).size();
        }
        this.avgIcpIterations = itersCounter / (double) facesCounter;
    }
    
    /**
     * Computes averaged face. It is supposed that faces used for the computation
     * are already superimposed (registered) with the template face.
     * 
     * @param initTempFaceId ID of the face used as an initial template for the computation of the averaged face
     * @param faceIDs IDs of faces that are used to compute averaged face
     */
    public void computeTemplateFace(String initTempFaceId, List<String> faceIDs) {
        if (faceIDs == null) {
            throw new IllegalArgumentException("faces");
        }
        
        this.templateFace = HumanFaceFactory.instance().getFace(initTempFaceId);
        
        HumanFaceFactory.instance().setStrategy(HumanFaceFactory.Strategy.MRU);
        AvgFaceConstructor constructor = new AvgFaceConstructor(this.templateFace.getMeshModel());
        for (String faceId: faceIDs) {
            //String faceId = HumanFaceFactory.instance().loadFace(new File(facePath));
            if (!this.templateFace.getId().equals(faceId)) { // skip the same face
                HumanFace face = HumanFaceFactory.instance().getFace(faceId);
                face.computeKdTree(false).accept(constructor);
            }
        }
        
        this.templateFace.setMeshModel(constructor.getAveragedMeshModel());
    }
    
    /**
     * Returns template face that can be used for linear time N:N distance computation.
     * 
     * @return template face.
     */
    public HumanFace getAvgTemplateFace() {
        return templateFace;
    }
    
    /**
     * Returns average number of ICP iterations per inspected face.
     * @return average number of ICP iterations per inspected face.
     */
    public double getAvgNumIcpIterations() {
        return this.avgIcpIterations;
    }
    
}
