package cz.fidentis.analyst.mesh.core;
import javax.vecmath.Vector3d;

/**
 *
 * @author Matej Lukeš
 */
//prerobenee
public class MeshPoint {
    protected Vector3d position, normal, texCoord;

    public MeshPoint(Vector3d position, Vector3d normal , Vector3d texCoord) {
        if (position == null)
            throw new NullPointerException("position cannot be null");

        this.position = position;
        this.normal = normal;
        this.texCoord = texCoord;
    }

    /**
     * @return normal
     */
    public Vector3d getNormal() {
        return normal;
    }

    /**
     * @return position
     */
    public Vector3d getPosition() {
        return position;
    }

    /**
     * @return texture coordinates
     */
    public Vector3d getTexCoord() {
        return texCoord;
    }
    
    /**
    * copy constructor of meshPoint
    * @param meshPoint copied meshPoint
    */
    public MeshPoint(MeshPoint meshPoint) {
        this.position = new Vector3d(meshPoint.position);
        if (meshPoint.normal != null) {
            this.normal = new Vector3d(meshPoint.normal);
        }
        if (meshPoint.texCoord != null) {
            this.texCoord = new Vector3d(meshPoint.texCoord);
        }
    }


    /**
     * @param obj compared object
     * @return true if positions, normals and texture coordinates are equal, false otherwise
     */
    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof MeshPoint))
            return false;

        MeshPoint meshPointObj = (MeshPoint) obj;
        return this.position.equals(meshPointObj.position);
    }

    @Override
    public int hashCode() {
        return position.hashCode(); 
    }        
    
    /**
     * returns new instance of MeshPoint with subtracted position
     *
     * @param subtrahend position to be subtracted
     * @return subtracted MeshPoint
     */
    public MeshPoint subtractPosition(MeshPoint subtrahend) {
        return MeshPoint.this.subtractPosition(subtrahend.position);
    }

    /**
     * returns new instance of MeshPoint with subtracted position
     *
     * @param subtrahend position to be subtracted
     * @return subtracted MeshPoint
     */
    public MeshPoint subtractPosition(Vector3d subtrahend) {
        Vector3d newPosition = new Vector3d(position);
        newPosition.sub(subtrahend);
        if (normal != null) {
            if (texCoord != null) 
                return new MeshPoint(new Vector3d(newPosition), new Vector3d(normal), new Vector3d(texCoord));
            else 
                return new MeshPoint(new Vector3d(newPosition), new Vector3d(normal), null);
        }
        return new MeshPoint(new Vector3d(newPosition), null, null);  
    }

    /**
     * returns new instance of MeshPoint with added position
     *
     * @param addend position to be added
     * @return added MeshPoint
     */
    public MeshPoint addPosition(MeshPoint addend) {
        return MeshPoint.this.addPosition(addend.position);
    }

    /**
     * returns new instance of MeshPoint with added position
     *
     * @param addend position to be added
     * @return added MeshPoint
     */
    public MeshPoint addPosition(Vector3d addend) {
        Vector3d newPosition = new Vector3d(position);
        newPosition.add(addend);
        if (normal != null) {
            if (texCoord != null) 
                return new MeshPoint(new Vector3d(newPosition), new Vector3d(normal), new Vector3d(texCoord));
            else 
                return new MeshPoint(new Vector3d(newPosition), new Vector3d(normal), null);
        }
        return new MeshPoint(new Vector3d(newPosition), null, null);
    }

    /*public MeshPoint addPosition(MeshPoint b) {
        return (new MeshPoint(new Vector3d(this.getPosition().x + b.getPosition().x,
                this.getPosition().y + b.getPosition().y, this.getPosition().z + b.getPosition().z),
                null, null));
    }*/
    
   /* public MeshPoint subtractPosition(MeshPoint b) {
        return (new MeshPoint(new Vector3d(this.getPosition().x - b.getPosition().x,
                this.getPosition().y - b.getPosition().y, this.getPosition().z - b.getPosition().z),
                null, null));
    }*/
    
    public MeshPoint multiplyPosition(double number) {
        if (normal != null) {
            if (texCoord != null) 
                return new MeshPoint(new Vector3d(this.getPosition().x * number,
                        this.getPosition().y * number, this.getPosition().z * number),
                        new Vector3d(normal), new Vector3d(texCoord));
            else
                return new MeshPoint(new Vector3d(this.getPosition().x * number,
                        this.getPosition().y * number, this.getPosition().z * number),
                        new Vector3d(normal), null);
        }
        return new MeshPoint(new Vector3d(this.getPosition().x * number,
                        this.getPosition().y * number, this.getPosition().z * number),
                        null, null);        
    }
    
    public MeshPoint dividePosition(double number) {
        if (normal != null) {
            if (texCoord != null) 
                return new MeshPoint(new Vector3d(this.getPosition().x / number, this.getPosition().y / number,
                        this.getPosition().z / number), new Vector3d(normal), new Vector3d(texCoord));
            else
                return new MeshPoint(new Vector3d(this.getPosition().x / number, this.getPosition().y / number,
                        this.getPosition().z / number), new Vector3d(normal), null);
        }
        return new MeshPoint(new Vector3d(this.getPosition().x / number, this.getPosition().y / number,
                this.getPosition().z / number), null, null);
    }
    
     /**
      * Returns the cross product of two points.
        @param meshPoint Second argument of the cross product operation.
     * @return 
        @returns Point3D representing the resulting vector.
     */
    public MeshPoint crossProduct(MeshPoint meshPoint) {
        if (normal != null) {
            if (texCoord != null) 
                return new MeshPoint(new Vector3d
                (this.position.y * meshPoint.position.z - this.position.z * meshPoint.position.y,
                this.position.z * meshPoint.position.x - this.position.x * meshPoint.position.z,
                this.position.x * meshPoint.position.y - this.position.y * meshPoint.position.x),
                new Vector3d(normal), new Vector3d(texCoord));
            else
                return new MeshPoint(new Vector3d
                (this.position.y * meshPoint.position.z - this.position.z * meshPoint.position.y,
                this.position.z * meshPoint.position.x - this.position.x * meshPoint.position.z,
                this.position.x * meshPoint.position.y - this.position.y * meshPoint.position.x),
                new Vector3d(normal), null);
        }
        return new MeshPoint(new Vector3d
                (this.position.y * meshPoint.position.z - this.position.z * meshPoint.position.y,
                this.position.z * meshPoint.position.x - this.position.x * meshPoint.position.z,
                this.position.x * meshPoint.position.y - this.position.y * meshPoint.position.x),
                null, null);
    }

    public double dotProduct(MeshPoint meshPoint) {
        return (this.position.x * meshPoint.position.x + this.position.y * meshPoint.position.y + this.position.z * meshPoint.position.z);
    }
    
    public double abs() {
        return Math.sqrt(this.getPosition().x * this.getPosition().x + 
                this.getPosition().y * this.getPosition().y + this.getPosition().z * this.getPosition().z);
    }
}
