package cz.fidentis.analyst.mesh.io;

import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshModel;
import cz.fidentis.analyst.symmetry.Triangle;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Locale;

/**
 * Utility class for exporting model to .obj file (vertices, normals and triangles)
 * 
 * @author Natalia Bebjakova
 */
public class MeshObjExporter {
    /**
     * Model to be exported
     */
    MeshModel model;
    
    /**
     *
     * @param model model to be exported.
     */
    public MeshObjExporter(MeshModel model) {
        this.model = model;
    }
    
    /**
     * Exports all facets of the model to .obj file
     * 
     * @param exportFile File to which model is exported
     * @throws IOException 
     */
    public void exportModelToObj(File exportFile) throws IOException {
        for(MeshFacet facet : model.getFacets()) {
            exportFacetToObj(facet, exportFile);
        }
    }

    /**
     * Exports facet to OBJ file.
     * It writes vertices, normals and triangles to file
     * 
     * @param facet Facet of the model to be exported, so far every model has one
     * @param exportFile file for exporting.
     * @throws java.io.IOException 
     */
    public void exportFacetToObj(MeshFacet facet, File exportFile) throws IOException {
        int formatIndex = exportFile.getName().lastIndexOf(".");
        String fileName; //name that is writen to file
        
        if (formatIndex < 0) {
            fileName = exportFile.getName();
        } else {
            fileName = exportFile.getName().substring(0, formatIndex);
        }

        exportFile = new File(exportFile.getParent() + File.separator + fileName + ".obj");
        
        try (FileWriter fstream = new FileWriter(exportFile);
        BufferedWriter out = new BufferedWriter(fstream);) {
           
            DecimalFormatSymbols otherSymbols = new DecimalFormatSymbols(Locale.getDefault());
            otherSymbols.setDecimalSeparator('.');  //dot as separator for decimal numbers
            DecimalFormat df = new DecimalFormat("#.####", otherSymbols);
             
            //writes vertices of the facet to file
            for (int j = 0; j < facet.getNumberOfVertices(); j++) {
                out.write("v " + df.format(facet.getVertices().get(j).getPosition().x) + " " 
                        + df.format(facet.getVertices().get(j).getPosition().y) + " " 
                        + df.format(facet.getVertices().get(j).getPosition().z) + "\n");
            }
            
            //detects if first vertex has normal
            boolean hasAllNormals = facet.getVertices().get(0).getNormal() != null;
            
            //writes normals if there are any 
            for (int i = 0; i < facet.getNumberOfVertices(); i++) {
                if(facet.getVertex(i).getNormal() != null) {
                out.write("vn " + df.format(facet.getVertices().get(i).getNormal().x) + " "
                        + df.format(facet.getVertices().get(i).getNormal().y) + " "
                        + df.format(facet.getVertices().get(i).getNormal().z) + "\n");
                }
            }
            
            //computes triangles of facet
            Triangle[] triangles = new Triangle[facet.getCornerTable().getSize() / 3];
            for (int i = 0; i < facet.getCornerTable().getSize(); i += 3) {
                Triangle t = new Triangle(facet.getCornerTable().getRow(i).getVertexIndex(),
                    facet.getCornerTable().getRow(i + 1).getVertexIndex(),
                    facet.getCornerTable().getRow(i + 2).getVertexIndex());
                triangles[(i / 3)] = t;
            }
            
            //writes triangles of facet
            for (Triangle triangle : triangles) {
                out.write("f ");
                if (facet.getVertex(triangle.getVertex1()).getNormal() != null && hasAllNormals) {
                    out.write((triangle.getVertex1() + 1) + "//" + (triangle.getVertex1() + 1) + " "
                            + (triangle.getVertex2() + 1) + "//" + (triangle.getVertex2() + 1) + " "
                            + (triangle.getVertex3() + 1) + "//" + (triangle.getVertex3() + 1) + "\n"); 
                } else {
                    out.write((triangle.getVertex1() + 1) + " " + (triangle.getVertex2() + 1) +
                            " " + (triangle.getVertex3() + 1) + "\n");   
                }
            }
            out.write("#" + triangles.length + " triangles" + "\n");
            out.write("\n");
            out.close();
            fstream.close();
        }
    }
}

