package cz.fidentis.analyst.newgui.swing;

import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseWheelEvent;
import javax.swing.SwingUtilities;

/**
 * 3D manipulation with mouse, using 3D movement from GeneralGLEventListener.
 * 
 * @author Richard Pajersky
 */
public class MouseRotationListener extends MouseAdapter {
    
    private final CanvasPanel canvas;
    
    private float lastX = 0;
    private float lastY = 0;
    
    private static double rotationSpeed = 0.4;
    private static double moveSpeed = 0.4;

    /**
     * Constructor.
     * 
     * @param canvas Canvas
     * @throws IllegalArgumentException if the canvas is {@code null}
     */
    public MouseRotationListener(CanvasPanel canvas) {
        if (canvas == null) {
            throw new IllegalArgumentException("canvas is nuull");
        }
        this.canvas = canvas;
    }

    /**
     * Left mouse button dragging rotates
     * Right mouse button dragging moves
     * Middle mouse button dragging resets rotation and zoom
     */
    @Override
    public void mouseDragged(MouseEvent evt) {
        //JOptionPane.showMessageDialog(null, evt.getButton());

        if (SwingUtilities.isLeftMouseButton(evt)) {
            double rotateX = -(lastY - evt.getY()) * rotationSpeed;
            double rotateY = +(lastX - evt.getX()) * rotationSpeed;
            if (Math.abs(rotateX) < Math.abs(rotateY)) {
                rotateX = 0;
            } else if (Math.abs(rotateY) < Math.abs(rotateX)) {
                rotateY = 0;
            }
            canvas.getScene().getCamera().rotate(rotateX, rotateY);
        }
        if (SwingUtilities.isRightMouseButton(evt)) {
            double moveX = -(lastX - evt.getX()) * moveSpeed;
            double moveY = -(lastY - evt.getY()) * moveSpeed;
            canvas.getScene().getCamera().move(moveX, moveY);
        }
        if (SwingUtilities.isMiddleMouseButton(evt)) {
            canvas.getScene().getCamera().initLocation();
        }
        lastX = evt.getX();
        lastY = evt.getY();
        canvas.getGLCanvas().display();
    }

    /**
     * Actualize mouse movement
     */
    @Override
    public void mouseMoved(MouseEvent e) {
        lastX = e.getX();
        lastY = e.getY();
    }

    /**
     * Zoom in or out based on mouse wheel movement
     */
    @Override
    public void mouseWheelMoved(MouseWheelEvent evt) {
        if (evt.getWheelRotation() > 0) {
            canvas.getScene().getCamera().zoomIn(-5 * evt.getWheelRotation());
        } else {
            canvas.getScene().getCamera().zoomOut(5 * evt.getWheelRotation());
        }
        canvas.getGLCanvas().display();
    }
    
    /**
     * Middle mouse button click resets rotation and zoom
     */
    @Override
    public void mouseClicked(MouseEvent evt) {
        if (SwingUtilities.isMiddleMouseButton(evt)) {
            canvas.getScene().getCamera().initLocation();
            canvas.getGLCanvas().display();
        }
    }
    
    public static double getRotationSpeed() {
        return rotationSpeed;
    }

    public static void setRotationSpeed(double rotationSpeed) {
        MouseRotationListener.rotationSpeed = rotationSpeed;
    }

    public static double getMoveSpeed() {
        return moveSpeed;
    }

    public static void setMoveSpeed(double moveSpeed) {
        MouseRotationListener.moveSpeed = moveSpeed;
    }

}
