package cz.fidentis.analyst.mesh.core;

import java.util.List;
import cz.fidentis.analyst.mesh.MeshVisitor;
import javax.vecmath.Vector3d;

/**
 * An ancapsulated mesh plate, i.e., multiple triangles sharing vertices.
 * Mesh facet is iterable (triangle as returned) and visitable by 
 * {@link cz.fidentis.analyst.mesh.MeshVisitor}s.
 *
 * @author Matej Lukes
 */
public interface MeshFacet extends Iterable<MeshTriangle> {
    
    /**
     * returns vertex of specified index
     *
     * @param index index of vertex
     * @return vertex
     */
    MeshPoint getVertex(int index);

    /**
     * adds vertex to MeshFacet
     *
     * @param point new vertex
     */
    void addVertex(MeshPoint point);

    /**
     * returns number of vertices in MeshFacet
     *
     * @return number of vertices
     */
    int getNumberOfVertices();

    /**
     * returns list of vertices in MeshFacet
     *
     * @return list if vertices
     */
    List<MeshPoint> getVertices();

    /**
     * returns Corner Table representing MeshFacet
     *
     * @return corner table
     */
    CornerTable getCornerTable();
    
    /**
     * Returns true if normals of vertices are calculated.
     * @return true if normals of vertices are calculated.
     */
    boolean hasVertexNormals();
    
    /**
     * Calculates normals of vertices from normals of triangles.
     */
    void calculateVertexNormals();
    
    /**
     * Returns number of triangles.
     * 
     * @return number of triangles
     */
    int getNumTriangles();
    
    /**
     * Returns triangles sharing the given mesh vertex.
     * 
     * @param vertexIndex Index of the mesh vertex
     * @return Triangles sharing the mesh vertex
     */
    List<MeshTriangle> getAdjacentTriangles(int vertexIndex);
    
    /**
     * Finds and returns a point lying at triangles around (sharing) the given mesh vertex
     * and being the closest to a 3D point.
     * 
     * @param point 3D point
     * @param vertexIndex Index of mesh vertex
     * @return The closest surface point near the mesh vertex being closest 
     * to the given 3D point
     */
    Vector3d getClosestAdjacentPoint(Vector3d point, int vertexIndex);
    
    /**
     * Returns the distance between a 3D point and triangles around (sharing)
     * the given mesh vertex. It is the distance between the 3D point and 
     * point found by the {@link MeshFacet#getClosestAdjacentPoint} method.
     * 
     * @param point 3D point
     * @param vertexIndex Index of mesh vertex
     * @return Distance
     */
    double curvatureDistance(Vector3d point, int vertexIndex);
    
    /**
     * Visits this facet.
     * 
     * @param visitor Visitor
     */
    void accept(MeshVisitor visitor);    
}
